// services/lobbyService.js
require('dotenv').config();
const blockService = require('./blockService');

/** Normaliza valores diversos para 'male' | 'female' | 'other' */
function normalizeGender(g) {
  if (!g) return 'other';
  const s = String(g).trim().toLowerCase();
  const maleSet = new Set(['male','m','masculino','homem','masc']);
  const femaleSet = new Set(['female','f','feminino','mulher','fem']);
  if (maleSet.has(s)) return 'male';
  if (femaleSet.has(s)) return 'female';
  return 'other';
}

class LobbyService {
  constructor() {
    this.rooms = new Map();
    this.countdownTimers = new Map();
    this.initializeRooms();
    this.LOBBY_COUNTDOWN_SECONDS = parseInt(process.env.LOBBY_COUNTDOWN_SECONDS || '5', 10);
  }

  initializeRooms() {
    const categories = [
      { id: 'heterossexual', name: 'Heterossexual' },
      { id: 'homossexual_masculino', name: 'Homossexual' },
      { id: 'homossexual_feminino', name: 'Homossexual' },
      { id: 'lgbt', name: 'LGBT+' }
    ];

    categories.forEach(category => {
      this.rooms.set(`${category.id}_sala1`, {
        id: `${category.id}_sala1`,
        name: `${category.name} - Sala 1 (2 pessoas)`,
        maxUsers: 2,
        category: category.id,
        users: new Map(),
        sockets: new Map()
      });
      this.rooms.set(`${category.id}_sala2`, {
        id: `${category.id}_sala2`,
        name: `${category.name} - Sala 2 (4 pessoas)`,
        maxUsers: 4,
        category: category.id,
        users: new Map(),
        sockets: new Map()
      });
      this.rooms.set(`${category.id}_sala3`, {
        id: `${category.id}_sala3`,
        name: `${category.name} - Sala 3 (6 pessoas)`,
        maxUsers: 6,
        category: category.id,
        users: new Map(),
        sockets: new Map()
      });
    });
  }

  async joinRoom(userData, roomId, socketId) {
    const room = this.rooms.get(roomId);
    if (!room) throw new Error('Sala não encontrada');

    const { userId, username } = userData;
    const gender = normalizeGender(userData.gender);
    const { sessionId } = userData;

    const existing = room.users.get(sessionId);

    if (!existing) {
      if (!this.canUserJoinRoom(room, gender)) {
        throw new Error('Sala já tem o máximo de participantes deste gênero');
      }
      room.users.set(sessionId, { userId, username, gender, socketId, sessionId });
    } else {
      // reconexão dentro do lobby
      existing.socketId = socketId;
      existing.gender = gender; // garante normalização
      existing.username = username ?? existing.username;
    }

    if (socketId) room.sockets.set(socketId, sessionId);

    if (room.users.size >= room.maxUsers) {
      if (room.category === 'heterossexual') {
        const maleCount = Array.from(room.users.values()).filter(u => u.gender === 'male').length;
        const femaleCount = Array.from(room.users.values()).filter(u => u.gender === 'female').length;
        if (maleCount === femaleCount) {
          return { status: 'ready', room, triggerCountdown: true };
        }
      } else {
        return { status: 'ready', room, triggerCountdown: true };
      }
    }

    return { status: 'joined', room };
  }

  async startCountdown(roomId, emitCallback) {
    const room = this.rooms.get(roomId);
    if (!room) return;

    const ready = await this.isRoomReady(room);
    if (!ready) return;

    this.stopCountdown(roomId);

    let countdown = Number.isFinite(this.LOBBY_COUNTDOWN_SECONDS) ? this.LOBBY_COUNTDOWN_SECONDS : 5;

    const tick = async () => {
      const stillReady = await this.isRoomReady(room);
      if (!stillReady) {
        this.stopCountdown(roomId);
        emitCallback('countdownCancelled', roomId);
        return;
      }

      emitCallback('countdownUpdate', roomId, { countdown });

      if (countdown <= 0) {
        this.stopCountdown(roomId);
        await this.transferToChat(roomId, emitCallback);
        return;
      }

      countdown -= 1;
      this.countdownTimers.set(roomId, setTimeout(tick, 1000));
    };

    this.countdownTimers.set(roomId, setTimeout(tick, 0));
  }

  stopCountdown(roomId) {
    const t = this.countdownTimers.get(roomId);
    if (t) {
      clearTimeout(t);
      this.countdownTimers.delete(roomId);
    }
  }

  async isRoomReady(room) {
    if (!room || room.users.size < room.maxUsers) return false;

    const users = Array.from(room.users.values());

    // Regras por categoria (com gênero normalizado)
    if (room.category === 'heterossexual') {
      const maleCount = users.filter(u => u.gender === 'male').length;
      const femaleCount = users.filter(u => u.gender === 'female').length;
      if (maleCount !== femaleCount) return false;
    } else if (room.category === 'homossexual_masculino') {
      if (users.some(u => u.gender !== 'male')) return false;
    } else if (room.category === 'homossexual_feminino') {
      if (users.some(u => u.gender !== 'female')) return false;
    }

    // Bloqueios
    const { hasConflict } = await blockService.hasAnyBlockingConflict(users);
    if (hasConflict) return false;

    return true;
  }

  async transferToChat(roomId, emitCallback) {
    const room = this.rooms.get(roomId);
    if (!room) return;

    const usersArray = Array.from(room.users.values()).filter(u => u.socketId);
    if (usersArray.length < room.maxUsers) {
      emitCallback('countdownCancelled', roomId);
      return;
    }

    const { hasConflict } = await blockService.hasAnyBlockingConflict(usersArray);
    if (hasConflict) {
      emitCallback('countdownCancelled', roomId);
      return;
    }

    const pairs = new Map();
    const paired = new Set();

    if (room.category === 'heterossexual') {
      const males = usersArray.filter(u => u.gender === 'male');
      const females = usersArray.filter(u => u.gender === 'female');
      for (let i = 0; i < Math.min(males.length, females.length); i++) {
        pairs.set(males[i].sessionId, females[i].sessionId);
        pairs.set(females[i].sessionId, males[i].sessionId);
        paired.add(males[i].sessionId);
        paired.add(females[i].sessionId);
      }
    } else if (room.category === 'homossexual_masculino' || room.category === 'homossexual_feminino') {
      for (let i = 0; i < usersArray.length; i += 2) {
        if (i + 1 < usersArray.length) {
          pairs.set(usersArray[i].sessionId, usersArray[i + 1].sessionId);
          pairs.set(usersArray[i + 1].sessionId, usersArray[i].sessionId);
          paired.add(usersArray[i].sessionId);
          paired.add(usersArray[i + 1].sessionId);
        }
      }
    } else {
      for (let i = 0; i < usersArray.length; i++) {
        const user1 = usersArray[i];
        if (paired.has(user1.sessionId)) continue;
        for (let j = i + 1; j < usersArray.length; j++) {
          const user2 = usersArray[j];
          if (!paired.has(user2.sessionId)) {
            pairs.set(user1.sessionId, user2.sessionId);
            pairs.set(user2.sessionId, user1.sessionId);
            paired.add(user1.sessionId);
            paired.add(user2.sessionId);
            break;
          }
        }
      }
    }

    room.pairs = pairs;

    usersArray.forEach(user => {
      const partnerSessionId = pairs.get(user.sessionId);
      const partnerInfo = usersArray.find(u => u.sessionId === partnerSessionId);
      if (!partnerInfo) return;

      const transferData = {
        roomId,
        users: usersArray.map(u => ({ ...u, userId: u.userId || null })),
        currentUser: { ...user, userId: user.userId },
        partner: { ...partnerInfo, userId: partnerInfo.userId }
      };

      emitCallback('redirectToChat', user.socketId, {
        action: 'redirect',
        url: `/chat?roomId=${roomId}&sessionId=${user.sessionId}`,
        data: transferData
      });
    });
  }

  handleChatJoined(sessionId, roomId) {
    const room = this.rooms.get(roomId);
    if (!room) return;

    const user = room.users.get(sessionId);
    if (user) user.inChat = true;

    const allDone = Array.from(room.users.values()).every(u => u.inChat || !u.socketId);
    if (allDone) {
      room.users.clear();
      room.sockets.clear();
      return true;
    }
    return false;
  }

  /**
   * Remove usuário de uma sala (por clique em "Sair" OU por desconexão explícita).
   * Retorna detalhes para o controller poder emitir eventos corretos para o lobby.
   */
  leaveRoom(socketId, roomId, sessionId) {
    const room = this.rooms.get(roomId);
    if (!room) return { roomUpdated: false, countdownCancelled: false, leftUser: null };

    // Identifica sessionId a partir do socketId, se necessário
    let sessId = sessionId;
    if (!sessId && socketId) {
      sessId = room.sockets.get(socketId);
    }
    if (!sessId) return { roomUpdated: false, countdownCancelled: false, leftUser: null };

    const user = room.users.get(sessId) || null;
    const leftUser = user ? { ...user } : null;

    if (socketId) room.sockets.delete(socketId);
    room.users.delete(sessId);

    let cancelled = false;
    if (this.countdownTimers.has(roomId)) {
      this.stopCountdown(roomId);
      cancelled = true;
    }
    return { roomUpdated: true, countdownCancelled: cancelled, leftUser };
  }

  /**
   * Desconexão de socket: varre salas, remove e acumula quem saiu para o controller emitir 'lobbyUserLeft'
   */
  handleUserDisconnect(socketId, emitCallback) {
    let roomChanged = false;
    const left = []; // [{ roomId, user }]

    this.rooms.forEach((room, roomId) => {
      const sessionId = room.sockets.get(socketId);
      if (sessionId) {
        const leaveResult = this.leaveRoom(socketId, roomId, sessionId);
        if (leaveResult.roomUpdated) {
          roomChanged = true;
          if (leaveResult.countdownCancelled) {
            emitCallback('countdownCancelled', roomId);
          }
          if (leaveResult.leftUser) {
            left.push({ roomId, user: leaveResult.leftUser });
          }
        }
      }
    });

    return { roomChanged, left };
  }

  canUserJoinRoom(room, genderRaw) {
    const gender = normalizeGender(genderRaw);
    const users = Array.from(room.users.values());

    if (room.category === 'heterossexual') {
      const maleCount = users.filter(u => u.gender === 'male').length;
      const femaleCount = users.filter(u => u.gender === 'female').length;
      const max = room.maxUsers / 2;
      if (gender === 'male') return maleCount < max;
      if (gender === 'female') return femaleCount < max;
      // outros não entram nas salas heterossexuais
      return false;
    } else if (room.category === 'homossexual_masculino') {
      return gender === 'male' && users.length < room.maxUsers;
    } else if (room.category === 'homossexual_feminino') {
      return gender === 'female' && users.length < room.maxUsers;
    }
    return users.length < room.maxUsers; // lgbt/geral
  }

  getPublicRooms() {
    const categories = new Map();

    Array.from(this.rooms.values()).forEach(room => {
      let categoryName;
      if (room.category === 'heterossexual') categoryName = 'Heterossexual';
      else if (room.category === 'homossexual_masculino') categoryName = 'Homossexual';
      else if (room.category === 'homossexual_feminino') categoryName = 'Homossexual';
      else categoryName = 'LGBT+';

      if (!categories.has(room.category)) {
        categories.set(room.category, { name: categoryName, rooms: [] });
      }

      const users = Array.from(room.users.values());
      const maleCount = users.filter(u => u.gender === 'male').length;
      const femaleCount = users.filter(u => u.gender === 'female').length;

      categories.get(room.category).rooms.push({
        id: room.id,
        name: room.name,
        currentUsers: room.users.size,
        maxUsers: room.maxUsers,
        maleCount,
        femaleCount,
        maxMale: room.category === 'heterossexual' ? room.maxUsers / 2 :
                 (room.category === 'homossexual_masculino' ? room.maxUsers : 0),
        maxFemale: room.category === 'heterossexual' ? room.maxUsers / 2 :
                   (room.category === 'homossexual_feminino' ? room.maxUsers : 0)
      });
    });

    return Array.from(categories.values());
  }

  generateSessionId() {
    return `sess_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
  }

  getRoomUsers(roomId) {
    const room = this.rooms.get(roomId);
    return room ? Object.fromEntries(room.users) : {};
  }

  async getBlockingStatus(roomId) {
    const room = this.rooms.get(roomId);
    if (!room) return { hasBlockingConflict: false };
    const users = Array.from(room.users.values());
    const { hasConflict } = await blockService.hasAnyBlockingConflict(users);
    return { hasBlockingConflict: hasConflict };
  }
}

module.exports = new LobbyService();
