const express = require('express');
const { body } = require('express-validator');
const { handleValidation } = require('../middlewares/validationMiddleware');
const router = express.Router();
const path = require('path');
const authController = require('../controllers/authController');
const upload = require('../middlewares/upload');
const galleryUpload = require('../middlewares/galleryUpload');
const authMiddleware = require('../middlewares/authMiddleware'); 

const registerRules = [
  body('username')
    .trim()
    .notEmpty().withMessage('Informe um nome de usuário.')
    .isLength({ min: 3, max: 50 }).withMessage('Usuário deve ter entre 3 e 50 caracteres.'),
  body('email')
    .trim()
    .notEmpty().withMessage('Informe um e-mail.')
    .isEmail().withMessage('E-mail inválido.'),
  body('password')
    .notEmpty().withMessage('Informe uma senha.')
    .isStrongPassword({
      minLength: 8,
      minLowercase: 1,
      minUppercase: 0,
      minNumbers: 1,
      minSymbols: 0,
      returnScore: false
    }).withMessage('A senha deve ter no mínimo 8 caracteres e incluir pelo menos 1 número.'),
  body('gender')
    .optional({ nullable: true })
    .isIn(['male', 'female', 'other']).withMessage('Gênero inválido.')
];

const loginRules = [
  body('email').notEmpty().withMessage('Informe um e-mail.').isEmail().withMessage('E-mail inválido.'),
  body('password').notEmpty().withMessage('Informe a senha.')
];

router.get('/login', authController.showLogin);
router.post('/login', loginRules, handleValidation, authController.login);

router.get('/register', authController.showRegister);
router.post('/register', registerRules, handleValidation, authController.register);

// novos:
router.post('/refresh', authController.refresh); // usa apenas cookie rt
router.post('/logout-all', authMiddleware.requireAuth, authController.logoutAll);

router.get('/logout', authController.logout);

router.get('/profile', authMiddleware.requireAuth, authController.showProfile);
router.get('/me', authMiddleware.requireAuth, authController.getUserData);
router.post('/remove-profile-picture', authMiddleware.requireAuth, authController.removeProfilePicture);
router.post('/update-profile', authMiddleware.requireAuth, upload.single('profilePicture'), authController.updateProfile);
router.get('/gallery-images', authMiddleware.requireAuth, authController.getGalleryImages);
router.delete('/gallery-image/:id', authMiddleware.requireAuth, authController.deleteGalleryImage);
router.post('/update-interests', authMiddleware.requireAuth, authController.updateInterests);
router.get('/profile-options', authMiddleware.requireAuth, authController.profileOptions);
router.get('/interests', authMiddleware.requireAuth, authController.listInterests);
router.get('/profile-completion', authMiddleware.requireAuth, authController.profileCompletion);


router.post('/upload-gallery-image',
  authMiddleware.requireAuth,
  galleryUpload.single('galleryImage'),
  authController.uploadGalleryImage
);

// Multi (até 12 por requisição)
router.post('/upload-gallery-images',
  authMiddleware.requireAuth,
  galleryUpload.array('galleryImages', 12),
  authController.uploadGalleryImages
);

module.exports = router;