const production = true;

/* ----------------------------- Helpers de API ------------------------------ */
async function refreshTokens() {
  const res = await fetch('/auth/refresh', { method: 'POST', credentials: 'include' });
  return res.ok;
}

async function api(url, opts = {}, retry = true) {
  const res = await fetch(url, { credentials: 'include', ...opts });
  if (res.status !== 401) return res;

  let data = {};
  try { data = await res.clone().json(); } catch { }

  if (retry && data?.code === 'TOKEN_EXPIRED') {
    const ok = await refreshTokens();
    if (ok) return fetch(url, { credentials: 'include', ...opts });
  }
  return res;
}

/* ----------------------------- Utils ------------------------------ */
function getCookie(name) {
  const value = `; ${document.cookie}`;
  const parts = value.split(`; ${name}=`);
  if (parts.length === 2) return parts.pop().split(';').shift();
  return null;
}
function normalizeGender(g) {
  if (!g) return 'other';
  const s = String(g).trim().toLowerCase();
  const maleSet = new Set(['male', 'm', 'masculino', 'homem', 'masc']);
  const femaleSet = new Set(['female', 'f', 'feminino', 'mulher', 'fem']);
  if (maleSet.has(s)) return 'male';
  if (femaleSet.has(s)) return 'female';
  return 'other';
}

async function fetchUnreadTotalWithFallback() {
  try {
    const r = await api('/chatFriends/unread-total');
    if (r.ok) {
      const d = await r.json();
      if (typeof d.total === 'number') return d.total;
    } else if (r.status !== 404) {
      console.warn('unread-total endpoint não OK:', r.status);
    }
  } catch { }
  try {
    const r = await api('/friendship/friends');
    if (!r.ok) return 0;
    const friends = await r.json();
    if (!Array.isArray(friends)) return 0;
    return friends.reduce((acc, f) => acc + (Number(f.unread_count) || 0), 0);
  } catch { return 0; }
}

async function refreshFriendsBadge() {
  const badge = document.getElementById('friendsBadge');
  if (!badge) return;
  const total = await fetchUnreadTotalWithFallback();
  if (total > 0) {
    badge.textContent = total > 99 ? '99+' : String(total);
    badge.style.display = 'inline-flex';
  } else {
    badge.style.display = 'none';
    badge.textContent = '0';
  }
}

async function fetchRequestsCount() {
  try {
    const r = await api('/friendship/requests/count');
    if (!r.ok) return 0;
    const d = await r.json().catch(() => ({}));
    return Number(d?.count || 0);
  } catch { return 0; }
}

async function refreshRequestsBadge() {
  const badge = document.getElementById('requestsBadge');
  if (!badge) return;
  const total = await fetchRequestsCount();
  if (total > 0) {
    badge.textContent = total > 99 ? '99+' : String(total);
    badge.style.display = 'inline-flex';
  } else {
    badge.style.display = 'none';
    badge.textContent = '0';
  }
}

/* ----------------------------- Boot ------------------------------ */
document.addEventListener('DOMContentLoaded', initDashboard);

async function initDashboard() {
  try {
    const res = await api('/auth/me');
    if (!res.ok) {
      location.href = '/auth/login';
      return;
    }
    const userData = await res.json();
    wireNav(userData);
    new RoomBrowser(userData);
    await refreshFriendsBadge();
    await refreshRequestsBadge();
    await renderProfileCompletion();
  } catch (err) {
    console.error('Erro ao iniciar dashboard:', err);
    location.href = '/auth/login';
  }
}

async function renderProfileCompletion() {
  const wrap = document.getElementById('profileCompletion');
  if (!wrap) return;

  // Evita flicker: começa oculto; só mostramos se estiver incompleto
  wrap.hidden = true;

  const bar = wrap.querySelector('.pc-bar');
  const fill = document.getElementById('pcFill');
  const num  = document.getElementById('pcPercent');
  const list = document.getElementById('pcMissing');
  const btn  = document.getElementById('pcGoProfile');

  try {
    const res = await api('/auth/profile-completion');
    if (!res.ok) throw new Error('Falha ao obter conclusão do perfil');
    const data = await res.json();

    const pct = Math.max(0, Math.min(100, Number(data.percent || 0)));
    const isComplete =
      pct >= 100 ||
      (data?.counts?.missing === 0) ||
      (data?.weights && data.weights.earned >= data.weights.total);

    // Se completo, removemos do DOM: nada aparece no dashboard
    if (isComplete) {
      wrap.remove();
      return;
    }

    // Incompleto: mostramos a seção e renderizamos conteúdo
    wrap.hidden = false;

    // Barra de progresso + ARIA
    fill.style.width = pct + '%';
    num.textContent = pct + '%';
    if (bar) {
      bar.setAttribute('aria-valuenow', String(pct));
      bar.setAttribute('aria-label', `Progresso do perfil: ${pct}%`);
    }

    // Lista do que falta (top 6)
    const missing = Array.isArray(data.missing) ? data.missing.slice(0, 6) : [];
    list.innerHTML = '';
    missing.forEach(item => {
      const li = document.createElement('li');
      li.innerHTML = `
        <i class="fa-regular fa-circle-xmark"></i>
        <span>${item.label}</span>
        ${item.hint ? `<span class="hint">— ${item.hint}</span>` : ''}
      `;
      list.appendChild(li);
    });

    btn.onclick = () => { location.href = '/auth/profile'; };
  } catch (e) {
    console.warn('profile-completion:', e);
    // Em caso de erro, silenciosamente não mostramos nada
    wrap.remove();
  }
}

/* ----------------------------- Navegação ------------------------------ */
function wireNav(userData) {
  document.getElementById('friendsListBtn')?.addEventListener('click', () => {
    location.href = '/friends/list?userId=' + encodeURIComponent(userData.id);
  });
  document.getElementById('profileBtn')?.addEventListener('click', () => {
    location.href = '/auth/profile?userId=' + encodeURIComponent(userData.id);
  });
  document.getElementById('historyBtn')?.addEventListener('click', () => {
    location.href = '/chat/historico/' + encodeURIComponent(userData.id);
  });
  document.getElementById('friendRequestsBtn')?.addEventListener('click', () => {
    const b = document.getElementById('requestsBadge');
    if (b) { b.style.display = 'none'; b.textContent = '0'; }
    location.href = '/friends/requests';
  });
  document.getElementById('blacklistBtn')?.addEventListener('click', () => {
    location.href = '/block/view';
  });
}

/* ----------------------------- RoomBrowser ------------------------------ */
class RoomBrowser {
  constructor(userData) {
    this.userData = userData;
    this.userGender = normalizeGender(userData.gender);
    this.roomsList = document.getElementById('roomsList');
    this.explorerBar = document.getElementById('explorerBar');

    this.socket = io({
      withCredentials: true,
      auth: { token: getCookie('token') }
    });

    this.setupLogoutButton();
    this.init();
  }

  setupLogoutButton() {
    const logoutBtn = document.getElementById('logoutBtn');
    logoutBtn?.addEventListener('click', async () => {
      try {
        const res = await api('/auth/logout', { method: 'GET' }, false);
        if (res.ok) {
          sessionStorage.removeItem('token');
          location.href = '/';
        } else {
          throw new Error('Falha ao fazer logout');
        }
      } catch (error) {
        console.error('Erro ao fazer logout:', error);
        alert('Erro ao tentar sair. Tente novamente.');
      }
    });
  }

  init() {
    this.setupSocket();
    this.fetchRooms();
    this.loadExplorerUsers();
  }

  /* ---------- Socket.IO ---------- */
  setupSocket() {
    this.socket.on('connect', () => {
      this.fetchRooms();
    });

    this.socket.on('connect_error', async (err) => {
      if (err?.message === 'TOKEN_EXPIRED') {
        const ok = await refreshTokens();
        if (ok) {
          this.socket.auth = { token: getCookie('token') };
          this.socket.connect();
        }
      }
    });

    this.socket.on('updateRooms', (rooms) => this.displayRooms(rooms));
    this.socket.on('new_message', async (message) => {
      if (String(message.receiver_id) === String(this.userData.id)) {
        await refreshFriendsBadge();
      }
    });
  }

  /* ---------- API: Rooms ---------- */
  async fetchRooms() {
    try {
      const res = await api('/lobby/rooms');
      if (!res.ok) throw new Error('Falha ao carregar salas');
      const payload = await res.json();
      const categories = Array.isArray(payload) ? payload : (payload.categories || []);
      this.displayRooms(categories);
    } catch (error) {
      console.error('Erro ao buscar salas:', error);
      this.roomsList.innerHTML = '<li>Não foi possível carregar as salas.</li>';
    }
  }

  /** Descobre o prefixo da categoria a partir do primeiro room.id */
  getCategoryIdFromGroup(group) {
    const firstId = group?.rooms?.[0]?.id || '';
    const m = firstId.match(/^(heterossexual|homossexual_masculino|homossexual_feminino|lgbt)_/i);
    return m ? m[1] : 'desconhecida';
  }

  displayRooms(categories) {
    this.roomsList.innerHTML = (categories?.length ? '' : '<li>Nenhuma sala disponível</li>');

    (categories || []).forEach(category => {
      const catId = this.getCategoryIdFromGroup(category);

      // Regras de visibilidade:
      if (catId === 'homossexual_masculino' && this.userGender !== 'male') return;
      if (catId === 'homossexual_feminino' && this.userGender !== 'female') return;

      // Cabeçalho da categoria
      const cat = document.createElement('li');
      cat.className = 'room-info';
      cat.style.fontWeight = '700';
      cat.style.background = '#f8fafc';
      cat.style.border = '1px solid var(--border)';
      cat.style.borderRadius = '10px';
      cat.style.padding = '10px 14px';
      cat.style.marginTop = '12px';
      cat.textContent = category.name || 'Geral';
      this.roomsList.appendChild(cat);

      (category.rooms || []).forEach(room => {
        const li = document.createElement('li');

        const info = document.createElement('div');
        info.className = 'room-info';
        info.innerHTML = `
          <div class="room-name">${room.name}</div>
          <div class="room-users">${room.currentUsers}/${room.maxUsers} usuários</div>
          ${room.maxMale > 0 ? `
            <div class="room-gender-info">
              <span class="gender-count">
                <span class="gender-icon male-icon"></span>
                ${room.maleCount}/${room.maxMale} homens
              </span>
            </div>` : ''}
          ${room.maxFemale > 0 ? `
            <div class="room-gender-info">
              <span class="gender-count">
                <span class="gender-icon female-icon"></span>
                ${room.femaleCount}/${room.maxFemale} mulheres
              </span>
            </div>` : ''}
        `;

        const button = document.createElement('button');
        button.className = 'join-btn';
        button.textContent = 'Entrar';

        const gender = this.userGender;
        let canJoin = false;

        if ((room.id || '').includes('heterossexual')) {
          canJoin = gender &&
            room.currentUsers < room.maxUsers &&
            ((gender === 'male' && room.maleCount < room.maxMale) ||
              (gender === 'female' && room.femaleCount < room.maxFemale));
        } else if ((room.id || '').includes('homossexual_masculino')) {
          canJoin = gender === 'male' && room.currentUsers < room.maxUsers;
        } else if ((room.id || '').includes('homossexual_feminino')) {
          canJoin = gender === 'female' && room.currentUsers < room.maxUsers;
        } else {
          canJoin = room.currentUsers < room.maxUsers;
        }

        button.disabled = !this.userData.username || !canJoin;
        button.addEventListener('click', () => this.joinRoom(room.id));

        li.appendChild(info);
        li.appendChild(button);
        this.roomsList.appendChild(li);
      });
    });
  }

  async joinRoom(roomId) {
    try {
      const res = await api(`/lobby/enter?roomId=${encodeURIComponent(roomId)}`, {
        headers: { 'Content-Type': 'application/json' }
      });
      const data = await res.json();

      if (!res.ok || !data.success) {
        throw new Error(data.message || 'Erro ao entrar na sala');
      }

      this.socket.emit('joinLobby', {
        roomId: data.roomId || roomId,
        sessionId: data.sessionId,
        userId: this.userData.id,
        username: this.userData.username,
        gender: this.userGender
      }, (ack) => {
        if (ack && ack.status === 'success') {
          location.href = `/lobby?roomId=${encodeURIComponent(roomId)}&sessionId=${encodeURIComponent(data.sessionId)}&userId=${encodeURIComponent(this.userData.id)}&username=${encodeURIComponent(this.userData.username)}&gender=${encodeURIComponent(this.userGender)}`;
        } else {
          throw new Error(ack?.message || 'Erro na conexão com a sala');
        }
      });

    } catch (error) {
      console.error('Erro ao entrar na sala:', error);
      alert('Erro: ' + (error.message || 'Não foi possível entrar na sala'));
      this.fetchRooms();
    }
  }

  /* ---------- Explorer ---------- */
  async loadExplorerUsers() {
    try {
      const res = await api('/explorer/users');
      if (!res.ok) throw new Error('Falha ao carregar Explorer');
      const data = await res.json();

      if (data.success && Array.isArray(data.users) && data.users.length > 0) {
        this.renderExplorerUsers(data.users);
        this.explorerBar.style.display = 'block';
      } else {
        this.explorerBar.style.display = 'none';
      }
    } catch (error) {
      console.error('Erro ao carregar usuários do Explorer:', error);
      this.explorerBar.style.display = 'none';
    }
  }

  renderExplorerUsers(users) {
    const container = document.getElementById('explorerUsers');
    container.innerHTML = '';
    users.forEach(u => {
      if (!u) return;
      const el = document.createElement('div');
      el.className = 'explorer-user';
      el.innerHTML = `
        <img src="${u.profilePicture || '/img/user_placeholder.png'}" alt="${u.username}">
        <span>${u.username}</span>
      `;
      el.addEventListener('click', () => {
        location.href = `/explorer/ver/${u.id}?idUser=${this.userData.id}`;
      });
      container.appendChild(el);
    });
  }
}
