// C:\src\Clientes\Datetime\public\js\friends.js
document.addEventListener('DOMContentLoaded', async () => {
  const listEl = document.getElementById('friendsList');
  const emptyEl = document.getElementById('emptyState');
  const searchInput = document.getElementById('searchInput');
  const searchBtn = document.getElementById('searchBtn');
  const clearBtn = document.getElementById('clearBtn');
  const reqCountBadge = document.getElementById('reqCount');

  let currentUserId = null;

  // --------- Auth/me para saber meu id (usado no socket) ---------
  try {
    const meRes = await fetch('/auth/me', { credentials: 'include' });
    if (meRes.ok) {
      const me = await meRes.json();
      currentUserId = me?.id ?? null;
    }
  } catch {}

  // --------- Socket.IO (opcionalmente, se presente) -----------
  let socket = null;
  if (typeof io !== 'undefined') {
    socket = io({
      auth: { token: document.cookie.split('token=')[1]?.split(';')[0] }
    });

    // Chegada de nova mensagem => se for *recebida* de um amigo, aumenta badge daquele card
    socket.on('new_message', (msg) => {
      if (!msg || !currentUserId) return;

      // Só aumenta badge se a mensagem veio *de* um amigo *para mim*
      if (String(msg.receiver_id) === String(currentUserId)) {
        const card = listEl.querySelector(`.friend-card[data-id="${msg.sender_id}"]`);
        if (!card) return;
        const badge = card.querySelector('[data-role="msg-badge"]');
        if (badge) {
          const n = Number(badge.textContent || 0) + 1;
          badge.textContent = String(n);
          badge.style.display = 'inline-block';
        }
      }
    });
  }

  // --------- Inicialização ---------
  await loadRequestsCount();
  await loadFriends();

  // --------- Busca ---------
  searchBtn.addEventListener('click', () => loadFriends(searchInput.value.trim()));
  searchInput.addEventListener('keydown', (e) => { if (e.key === 'Enter') loadFriends(searchInput.value.trim()); });
  clearBtn.addEventListener('click', () => { searchInput.value = ''; loadFriends(); });

  // ================================
  //            FUNÇÕES
  // ================================

  async function loadRequestsCount() {
    try {
      const r = await fetch('/friendship/requests/count', { credentials: 'include' });
      if (!r.ok) throw new Error('Falha ao buscar contagem');
      const data = await r.json();
      reqCountBadge.textContent = data.count ?? 0;
    } catch (e) {
      console.warn('Erro contagem solicitações:', e.message);
      reqCountBadge.textContent = '0';
    }
  }

  async function loadFriends(searchTerm = '') {
    try {
      listEl.innerHTML = spinnerRow();
      emptyEl.style.display = 'none';

      const qs = searchTerm ? `?search=${encodeURIComponent(searchTerm)}` : '';
      const r = await fetch(`/friendship/friends${qs}`, {
        credentials: 'include',
        headers: { 'Accept': 'application/json' }
      });
      if (!r.ok) throw new Error('Falha ao carregar amigos');
      const friends = await r.json();

      if (!Array.isArray(friends) || friends.length === 0) {
        listEl.innerHTML = '';
        emptyEl.style.display = 'block';
        return;
      }
      renderFriends(friends);
    } catch (e) {
      console.error(e);
      listEl.innerHTML = errorRow(e.message);
    }
  }

  function renderFriends(items) {
    listEl.innerHTML = items.map(f => friendCard(f)).join('');

    // Botão: ver perfil
    listEl.querySelectorAll('[data-action="profile"]').forEach(btn => {
      btn.addEventListener('click', () => {
        window.location.href = `/amigos/ver/${btn.dataset.id}`;
      });
    });

    // Botão: chat (marca como lidas e navega)
    listEl.querySelectorAll('[data-action="chat"]').forEach(btn => {
      btn.addEventListener('click', async () => {
        const id = btn.dataset.id;
        try {
          await fetch(`/chatFriends/mark-read/${id}`, { method: 'POST', credentials: 'include' });
        } catch {}
        window.location.href = `/chatFriends/room?id_friend=${id}`;
      });
    });
  }

  function friendCard(f) {
    // classes alinham com friends.css (.friend-card, .friend-avatar, etc)
    const pic = f.profile_picture || '/img/user_placeholder.png';
    const since = f.friendship_date ? formatDate(f.friendship_date) : '';
    const unread = Number(f.unread_count || 0);
    const badgeStyle = unread > 0 ? 'display:inline-block;' : 'display:none;';

    return `
      <div class="friend-card" data-id="${f.id}">
        <img src="${pic}" class="friend-avatar" alt="${escapeHtml(f.username)}"/>
        <div class="friend-info">
          <div class="friend-name">${escapeHtml(f.username)}</div>
          <div class="friend-meta"><i class="fas fa-calendar-day"></i> Amigos desde: ${since}</div>
        </div>
        <div class="friend-actions">
          <button class="icon-btn" title="Ver perfil" data-action="profile" data-id="${f.id}">
            <i class="fas fa-user"></i>
          </button>
          <button class="icon-btn" title="Mensagens" data-action="chat" data-id="${f.id}">
            <i class="fas fa-comment-dots"></i>
            <span class="badge" data-role="msg-badge" style="${badgeStyle}">${unread}</span>
          </button>
        </div>
      </div>
    `;
  }

  // ================================
  //            HELPERS
  // ================================
  function formatDate(iso) {
    try {
      const d = new Date(iso);
      return d.toLocaleDateString('pt-BR', { year: 'numeric', month: '2-digit', day: '2-digit' });
    } catch { return '--/--/----'; }
  }

  function spinnerRow() {
    return `<div class="friend-card"><i class="fas fa-spinner fa-spin"></i>&nbsp;Carregando...</div>`;
    // Nota: usa .friend-card para manter consistência visual
  }

  function errorRow(msg) {
    return `<div class="friend-card" style="border:1px solid #fecaca;color:#991b1b;">
      <i class="fas fa-exclamation-triangle"></i>&nbsp;${escapeHtml(msg || 'Erro')}
    </div>`;
  }

  function escapeHtml(s) {
    return String(s).replace(/[&<>"'`=\/]/g, c => ({
      '&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;',
      "'": '&#39;', '/': '&#x2F;', '`': '&#x60;', '=': '&#x3D;'
    }[c]));
  }
});
