async function fetchWithRefresh(input, init = {}) {
  const res = await fetch(input, { credentials: 'include', ...init });
  if (res.status !== 401) return res;
  const r = await fetch('/auth/refresh', { method: 'POST', credentials: 'include' });
  if (!r.ok) return res;
  return fetch(input, { credentials: 'include', ...init });
}

document.addEventListener('DOMContentLoaded', loadRequests);

function escapeHtml(s = '') {
  return String(s).replace(/[&<>"']/g, m => (
    { '&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;' }[m]
  ));
}

function formatDate(dt) {
  try {
    const d = new Date(dt);
    if (Number.isNaN(d.getTime())) return '--/--/---- --:--';
    const date = d.toLocaleDateString('pt-BR', { day: '2-digit', month: '2-digit', year: 'numeric' });
    const time = d.toLocaleTimeString('pt-BR', { hour: '2-digit', minute: '2-digit' });
    return `${date} ${time}`;
  } catch { return '--/--/---- --:--'; }
}

async function loadRequests() {
  const list = document.getElementById('requestsList');
  try {
    const res = await fetchWithRefresh('/friendship/requests');
    if (!res.ok) throw new Error(await res.text() || `Erro HTTP ${res.status}`);
    const requests = await res.json();

    if (!Array.isArray(requests) || !requests.length) {
      list.innerHTML = `<div class="empty">Nenhuma solicitação pendente</div>`;
      return;
    }

    list.innerHTML = requests.map(req => {
      const name = escapeHtml(req.username || 'Usuário');
      const when = formatDate(req.created_at);
      const avatar = escapeHtml(req.profile_picture || '/img/user_placeholder.png');
      const id = Number(req.id);
      return `
        <article class="request-item" data-id="${id}">
          <div class="request-left">
            <img class="request-avatar" src="${avatar}" alt="${name}">
            <div>
              <div class="request-name">${name}</div>
              <div class="small">Enviado em: ${when}</div>
            </div>
          </div>
          <div class="request-actions">
            <button class="btn btn-success" data-action="accept" data-id="${id}">
              <i class="fa-solid fa-check"></i> Aceitar
            </button>
            <button class="btn btn-danger" data-action="reject" data-id="${id}">
              <i class="fa-solid fa-xmark"></i> Rejeitar
            </button>
          </div>
        </article>
      `;
    }).join('');

    // bind actions
    list.querySelectorAll('button[data-action]').forEach(btn => {
      btn.addEventListener('click', () => handleRequest(btn.dataset.id, btn.dataset.action, btn.closest('.request-item')));
    });
  } catch (e) {
    list.innerHTML = `<div class="empty empty--error">Erro: ${escapeHtml(e.message || 'Falha ao carregar')}</div>`;
  }
}

async function handleRequest(requestId, action, node) {
  const buttons = node.querySelectorAll('button');
  const prev = Array.from(buttons, b => b.innerHTML);
  try {
    buttons.forEach(b => { b.disabled = true; b.innerHTML = `<i class="fa-solid fa-spinner fa-spin"></i> Processando`; });

    const res = await fetchWithRefresh(`/friendship/requests/${requestId}`, {
      method: 'PUT',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ action }) // 'accept' | 'reject'
    });
    if (!res.ok) throw new Error(await res.text() || 'Falha ao processar');

    node.remove();
    if (!document.querySelector('.request-item')) {
      document.getElementById('requestsList').innerHTML = `<div class="empty">Nenhuma solicitação pendente</div>`;
    }
  } catch (e) {
    alert(e.message || 'Erro');
    buttons.forEach((b, i) => { b.disabled = false; b.innerHTML = prev[i]; });
  }
}
