// C:\src\Clientes\Datetime\public\js\friendProfile.js
(function () {
  // ===== Util =====
  function getCookie(name) {
    const escaped = name.replace(/([.*+?^${}()|[\]\\])/g, '\\$1');
    const m = document.cookie.match(new RegExp('(?:^|; )' + escaped + '=([^;]*)'));
    return m ? decodeURIComponent(m[1]) : undefined;
  }
  async function parseJsonSafe(res) {
    const ct = (res.headers?.get?.('content-type') || '').toLowerCase();
    if (!ct.includes('application/json')) {
      const t = await res.text().catch(() => '');
      throw new Error(`Resposta não-JSON (${res.status}). ${t.slice(0, 140)}`);
    }
    return res.json();
  }
  const fmt = {
    age: (v) => {
      if (!v) return '—';
      const d = new Date(v);
      if (isNaN(d)) return '—';
      const now = new Date();
      let a = now.getFullYear() - d.getFullYear();
      const m = now.getMonth() - d.getMonth();
      if (m < 0 || (m === 0 && now.getDate() < d.getDate())) a--;
      return `${a} anos`;
    },
    cm: (v) => v ? `${v} cm` : '—',
    kg: (v) => v ? `${v} kg` : '—',
    gender: (g) => g === 'male' ? 'Masculino' : g === 'female' ? 'Feminino' : 'Não informado',
    pill: (v) => (v && String(v).trim()) ? String(v).trim() : ''
  };

  // Catálogo (labels) — espelha o profile.js para exibir bonitinho
  const LABELS = {
    relationship_goals: {
      amizade: 'Amizade', casual: 'Casual', namoro: 'Namoro', casamento: 'Casamento',
      networking: 'Networking', parceria: 'Parceria criativa', nao_sei: 'Não sei ainda'
    },
    relationship_type: {
      monogamico: 'Monogâmico', aberto: 'Aberto', poliamor: 'Poliamor', discreto: 'Discreto', nao_sei: 'Não sei ainda'
    },
    has_children: {
      tenho_nao_quero_mais: 'Tenho, e não quero mais um.', tenho: 'Tenho', tenho_quero_mais: 'Tenho e quero mais.',
      nao_tenho_nao_quero: 'Não tenho, e não quero', nao_tenho_aberto: 'Não tenho mas estou aberto(a)', prefiro_nao_dizer: 'Prefiro não dizer'
    },
    drinks: { nao: 'Não', socialmente: 'Socialmente', raramente: 'Raramente', frequentemente: 'Frequentemente' },
    smokes: { nao: 'Não', ocasional: 'Ocasional', vape: 'Vape', sim: 'Sim' },
    political_orientation: {
      esquerda: 'Esquerda', centro_esquerda: 'Centro-esquerda', centro: 'Centro', centro_direita: 'Centro-direita',
      direita: 'Direita', libertario: 'Libertário', independente: 'Independente', prefiro_nao_dizer: 'Prefiro não dizer'
    },
    religion: {
      catolico: 'Católico(a)', evangelico: 'Evangélico(a)', espirita: 'Espírita', umbanda_candomble: 'Umbanda/Candomblé',
      judaico: 'Judaico(a)', muculmano: 'Muçulmano(a)', budista: 'Budista', agnostico: 'Agnóstico(a)', ateu: 'Ateu/Atéia',
      outra: 'Outra', prefiro_nao_dizer: 'Prefiro não dizer'
    },
    kitchen_persona: {
      chef_fim_de_semana: 'Chef de fim de semana', mestre_microondas: 'Mestre do micro-ondas',
      rei_da_airfryer: 'Rei/Rainha da airfryer', tempero_da_vovo: 'Tempero da vovó', queima_ate_gelo: 'Queima até gelo'
    },
    diet_style: {
      conto_macros: 'Conto macros', low_carb: 'Low-carb quando dá', vegetariano: 'Vegetariano(a)', vegano: 'Vegano(a)',
      onivoro_feliz: 'Onívoro(a) feliz', segunda_sem_carne: 'Só segunda sem carne',
      jejum_intermitente: 'Jejum intermitente', nao_faco_ideia: 'Não faço ideia 😅'
    },
    pets: {
      team_dogs: 'Team dogs', team_cats: 'Team cats', amo_todos: 'Amo todos', alergia_mas_tento: 'Tenho alergia mas tento',
      peixes_e_plantas: 'Peixes e plantas', prefiro_plantas: 'Prefiro plantas'
    },
    coffee: {
      sempre: 'Sempre!', com_acucar: 'Só com açúcar', com_leite: 'Com leite', descafeinado: 'Descafeinado',
      cha: 'Chá, por favor', nao_curto: 'Não curto'
    },
    sports_role: {
      atleta_dedicado: 'Atleta dedicado(a)', torcedor_de_sofa: 'Torcedor(a) de sofá', domingo_no_parque: 'Domingo no parque',
      marombeiro: 'Marombeiro(a)', yoga_pilates: 'Yoga/Pilates', corrida: 'Corrida', ciclista: 'Ciclista'
    },
    party_style: {
      dj_improvisado: 'DJ improvisado', guardiao_da_bolsa: 'Guardião(ã) da bolsa', dono_da_roda: 'Dono(a) da roda de conversa',
      primeiro_na_pista: 'Primeiro(a) na pista', volta_com_batata: 'Some e volta com batata', fotografo: 'Fotógrafo(a) do rolê',
      deus_me_livre: 'Deus me livre', prefiro_nem_ir: 'Prefiro nem ir'
    },
    gangster_persona: {
      estrategista: 'O estrategista', motorista_fuga: 'O motorista de fuga', hacker: 'O hacker',
      pacificador: 'O pacificador', esquece_senha: 'O que esquece a senha'
    }
  };

  // Map de ícones para redes
  const SOCIAL_ICON = {
    'Instagram': 'fa-brands fa-instagram',
    'X/Twitter': 'fa-brands fa-x-twitter',
    'LinkedIn': 'fa-brands fa-linkedin',
    'Facebook': 'fa-brands fa-facebook',
    'TikTok': 'fa-brands fa-tiktok',
    'YouTube': 'fa-brands fa-youtube',
    'Site/Portfólio': 'fa-solid fa-globe',
    'Outro': 'fa-solid fa-link'
  };

  // ===== STATE =====
  let socket = null;
  let INTEREST_CATALOG = []; // [{slug,label}]

  document.addEventListener('DOMContentLoaded', init);

  async function init() {
    const parts = window.location.pathname.split('/').filter(Boolean);
    const friendId = parts[parts.length - 1];

    // Socket (se disponível)
    try {
      if (typeof io !== 'undefined') {
        socket = io({ auth: { token: getCookie('token') } });
      }
    } catch (e) {
      console.warn('Socket.IO indisponível no perfil:', e);
    }

    try {
      // catálogo p/ mapear slugs -> labels
      await loadInterestCatalog();

      // Perfil
      const profileResponse = await fetch(`/friendship/view/${friendId}`, { credentials: 'include' });
      if (!profileResponse.ok) throw new Error('Erro ao carregar perfil');
      const friendData = await parseJsonSafe(profileResponse);

      renderProfile(friendData);
      await loadGalleryImages(friendId);
      await setupBlockActions(friendId);

      const unfriendBtn = document.getElementById('unfriendBtn');
      if (unfriendBtn) unfriendBtn.onclick = () => handleUnfriend(friendId);

      await refreshChatBadge(friendId);
      const msgBtn = document.getElementById('messageBtn');
      if (msgBtn) {
        msgBtn.onclick = async () => {
          const badge = document.getElementById('chatBadge');
          if (badge) { badge.style.display = 'none'; badge.textContent = '0'; }
          try { await fetch(`/chatFriends/mark-read/${friendId}`, { method: 'POST', credentials: 'include' }); } catch { }
          window.location.href = `/chatFriends/room?id_friend=${friendId}`;
        };
      }

      if (socket) {
        socket.on('new_message', (message) => {
          if (String(message.sender_id) === String(friendId)) bumpChatBadge();
        });
      }

    } catch (error) {
      console.error('Erro:', error);
      alert(error.message || 'Erro ao carregar perfil do amigo');
      window.location.href = '/friends';
    }
  }

  async function loadInterestCatalog() {
    try {
      const r = await fetch('/auth/interests', { credentials: 'include' });
      if (!r.ok) throw 0;
      const data = await parseJsonSafe(r);
      INTEREST_CATALOG = Array.isArray(data.items) ? data.items : [];
    } catch { INTEREST_CATALOG = []; }
  }

  // ===== RENDER =====
  function renderProfile(d) {
    // Header
    setText('username', d.username || 'Não informado');
    setText('userGender', fmt.gender(d.gender));
    setSrc('profilePicture', d.profile_picture || '/img/user_placeholder.png');

    // pills
    setPill('pronounsText', fmt.pill(d.pronouns));
    setPill('occupationText', fmt.pill(d.occupation));
    const loc = [fmt.pill(d.city), fmt.pill(d.state)].filter(Boolean).join(' / ');
    setPill('locationText', loc);

    // Sobre
    setText('bioText', d.bio?.trim() ? d.bio : '—');

    // Básico
    setText('birthDate', fmt.age(d.birth_date));
    const label = document.getElementById('birthDate')?.previousElementSibling;
    if (label && label.tagName === 'LABEL') label.textContent = 'Idade';
    setText('height', fmt.cm(d.height));
    setText('weight', fmt.kg(d.weight));

    // Relacionamento
    setLabel('relationship_goals', LABELS.relationship_goals, d.relationship_goals);
    setLabel('relationship_type', LABELS.relationship_type, d.relationship_type);
    setLabel('has_children', LABELS.has_children, d.has_children);
    setLabel('drinks', LABELS.drinks, d.drinks);
    setLabel('smokes', LABELS.smokes, d.smokes);
    setLabel('political_orientation', LABELS.political_orientation, d.political_orientation);
    setLabel('religion', LABELS.religion, d.religion);

    // Preferências
    setLabel('kitchen_persona', LABELS.kitchen_persona, d.kitchen_persona);
    setLabel('diet_style', LABELS.diet_style, d.diet_style);
    setLabel('pets', LABELS.pets, d.pets);
    setLabel('coffee', LABELS.coffee, d.coffee);
    setLabel('sports_role', LABELS.sports_role, d.sports_role);
    setLabel('party_style', LABELS.party_style, d.party_style);
    setLabel('gangster_persona', LABELS.gangster_persona, d.gangster_persona);

    // Tags
    renderTags('withYouTags', Array.isArray(d.with_you) ? d.with_you : []);
    renderTags('perfectDayTags', Array.isArray(d.perfect_day) ? d.perfect_day : []);

    // Interesses (slugs -> label do catálogo)
    const labels = (Array.isArray(d.interests) ? d.interests : [])
      .map(slug => INTEREST_CATALOG.find(i => i.slug === slug)?.label || slug);
    renderTags('interestsTags', labels);

    // Sociais (normalizados pelo backend)
    renderSocials('socialLinksList', Array.isArray(d.social_links) ? d.social_links : []);
  }

  function setText(id, text) { const el = document.getElementById(id); if (el) el.textContent = text; }
  function setSrc(id, url) { const el = document.getElementById(id); if (el) el.src = url; }
  function setPill(id, text) {
    const el = document.getElementById(id);
    if (!el) return;
    if (text) { el.textContent = text; el.style.display = 'inline-block'; }
    else { el.textContent = ''; el.style.display = 'none'; }
  }
  function setLabel(id, dict, value) {
    const el = document.getElementById(id); if (!el) return;
    el.textContent = value ? (dict?.[value] || value) : '—';
  }
  function renderTags(containerId, arr) {
    const c = document.getElementById(containerId);
    if (!c) return;
    c.innerHTML = '';
    if (!arr.length) { c.innerHTML = '<span class="muted">—</span>'; return; }
    arr.forEach(t => {
      const b = document.createElement('span');
      b.className = 'tag';
      b.innerHTML = `<i class="fa-solid fa-tag"></i> ${escapeHtml(t)}`;
      c.appendChild(b);
    });
  }
  function renderSocials(containerId, items) {
    const box = document.getElementById(containerId);
    if (!box) return;
    box.innerHTML = '';
    if (!items.length) { box.innerHTML = '<span class="muted">—</span>'; return; }

    items.forEach(({ network, url }) => {
      if (!url) return;
      const wrap = document.createElement('div');
      wrap.className = 'social-item';
      const icon = SOCIAL_ICON[network] || 'fa-solid fa-link';
      const safeUrl = trySanitizeUrl(url);
      wrap.innerHTML = `
        <i class="${icon}"></i>
        <div class="social-text">
          <div class="muted" style="font-size:.85rem">${escapeHtml(network || 'Link')}</div>
          <a href="${safeUrl}" target="_blank" rel="noopener">${escapeHtml(url)}</a>
        </div>
      `;
      box.appendChild(wrap);
    });
  }
  function trySanitizeUrl(u) {
    try {
      const url = new URL(u, window.location.origin);
      const proto = url.protocol.toLowerCase();
      if (proto === 'http:' || proto === 'https:') return url.toString();
    } catch { }
    return '#';
  }
  function escapeHtml(s) {
    return String(s).replace(/[&<>"'`]/g, c => ({ '&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#39;', '`': '&#96;' }[c]));
  }

  // ===== GALERIA =====
  async function loadGalleryImages(friendId) {
    const galleryGrid = document.getElementById('galleryGrid');
    try {
      galleryGrid.innerHTML = '<div class="muted" style="padding:16px"><i class="fas fa-spinner fa-spin"></i> Carregando galeria...</div>';

      const response = await fetch(`/friendship/galeria/${friendId}`, { credentials: 'include' });
      if (!response.ok) {
        if (response.status === 403) {
          galleryGrid.innerHTML = `<div class="muted" style="padding:16px"><i class="fas fa-lock"></i> Galeria indisponível para você.</div>`;
          return;
        }
        let message = 'Erro ao carregar galeria';
        try { const err = await response.json(); message = err.message || err.error || message; } catch { }
        throw new Error(message);
      }
      const data = await parseJsonSafe(response);
      renderGallery(data.images || []);
    } catch (error) {
      console.error('Erro ao carregar galeria:', error);
      galleryGrid.innerHTML = `<div class="muted" style="padding:16px"><i class="fas fa-exclamation-circle"></i> ${error.message || 'Não foi possível carregar a galeria'}</div>`;
    }
  }
  function renderGallery(images) {
    const grid = document.getElementById('galleryGrid');
    grid.innerHTML = '';
    if (!images || !images.length) {
      grid.innerHTML = `<div class="muted" style="padding:16px"><i class="fas fa-images"></i> Nenhuma imagem na galeria</div>`;
      return;
    }
    for (const image of images) {
      const item = document.createElement('div');
      item.className = 'gallery-item';
      item.innerHTML = `<img src="${escapeHtml(image.image_path)}" alt="Imagem da galeria" loading="lazy">`;
      grid.appendChild(item);
    }
  }

  // ===== BLOQUEIO / AMIZADE =====
  async function setupBlockActions(friendId) {
    const r = await fetch(`/friendship/is-blocked/${friendId}`, { credentials: 'include' });
    if (!r.ok) return;
    const { isBlocked } = await r.json().catch(() => ({}));
    const blockBtn = document.getElementById('blockUserBtn');
    if (!blockBtn) return;

    if (isBlocked) {
      blockBtn.innerHTML = '<i class="fas fa-ban"></i> Desbloquear Usuário';
      blockBtn.onclick = () => handleUnblockAction(friendId);
    } else {
      blockBtn.innerHTML = '<i class="fas fa-ban"></i> Bloquear Usuário';
      blockBtn.onclick = () => handleBlockAction(friendId);
    }
  }
  async function handleBlockAction(friendId) {
    if (!confirm('Bloquear este usuário? Isso removerá amizade e solicitações.')) return;
    try {
      const r = await fetch('/friendship/blocks', {
        method: 'POST', credentials: 'include',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ targetId: friendId })
      });
      const data = await r.json().catch(() => ({}));
      if (!r.ok) throw new Error(data?.error || data?.message || 'Falha ao bloquear');
      alert(data?.message || 'Usuário bloqueado com sucesso');
      window.location.href = '/friends';
    } catch (e) { alert(`Erro ao bloquear: ${e.message}`); }
  }
  async function handleUnblockAction(friendId) {
    if (!confirm('Deseja desbloquear este usuário?')) return;
    try {
      const r = await fetch(`/friendship/blocks/${friendId}`, { method: 'DELETE', credentials: 'include' });
      const data = await r.json().catch(() => ({}));
      if (!r.ok) throw new Error(data?.error || data?.message || 'Falha ao desbloquear');
      alert(data?.message || 'Usuário desbloqueado com sucesso');
      location.reload();
    } catch (e) { alert(`Erro ao desbloquear: ${e.message}`); }
  }
  async function handleUnfriend(friendId) {
    if (!confirm('Tem certeza que deseja desfazer a amizade?')) return;
    try {
      const r = await fetch(`/friendship/friends/${friendId}`, { method: 'DELETE', credentials: 'include' });
      if (!r.ok) { const t = await r.text().catch(() => ''); throw new Error(t || 'Falha ao desfazer amizade'); }
      alert('Amizade desfeita.');
      window.location.href = '/friends';
    } catch (e) { alert(`Erro ao desfazer amizade: ${e.message}`); }
  }

  // ===== BADGE DE CHAT =====
  async function refreshChatBadge(friendId) {
    try {
      const r = await fetch(`/chatFriends/unread/${friendId}`, { credentials: 'include' });
      const data = await r.json().catch(() => ({}));
      const unread = Number(data?.unread || 0);
      const badge = document.getElementById('chatBadge');
      if (!badge) return;
      if (unread > 0) { badge.textContent = unread; badge.style.display = 'inline-flex'; }
      else { badge.style.display = 'none'; badge.textContent = '0'; }
    } catch { }
  }
  function bumpChatBadge() {
    const badge = document.getElementById('chatBadge');
    if (!badge) return;
    const n = Number(badge.textContent || 0) + 1;
    badge.textContent = String(n);
    badge.style.display = 'inline-flex';
  }
})();
