class ChatClient {
  constructor() {
    this.socket = null;
    this.state = {
      roomId: null,
      sessionId: null,
      socketId: null,
      currentUser: null,
      partner: null,
      currentPartner: null,
      pastPartners: [],
      users: [],
      partnerOnline: true
    };

    this.init();
  }

  /* ===================== Boot ===================== */
  init() {
    this.$ = (sel) => document.querySelector(sel);
    this.parseUrlParams();
    this.setupConnection();
    this.setupEventListeners();
  }

  parseUrlParams() {
    const urlParams = new URLSearchParams(window.location.search);
    const roomId = urlParams.get('roomId');
    const sessionId = urlParams.get('sessionId');

    const raw = sessionStorage.getItem('chatTransferData') || localStorage.getItem('chatTransferBackup');
    if (!raw) return;

    try {
      const data = JSON.parse(raw);
      if (!data?.data?.roomId || !data?.data?.currentUser?.sessionId) throw new Error('Dados incompletos');

      if (roomId && roomId !== data.data.roomId) throw new Error('Inconsistência no roomId');
      if (sessionId && sessionId !== data.data.currentUser.sessionId) throw new Error('Inconsistência no sessionId');

      this.state = {
        ...this.state,
        roomId: data.data.roomId,
        sessionId: data.data.currentUser.sessionId,
        socketId: data.data.currentUser.socketId,
        currentUser: data.data.currentUser,
        users: data.data.users || [],
        partner: data.data.partner || null,
        currentPartner: data.data.partner || null
      };

      // (removido) userName no header — apenas checa se existir
      const userNameEl = this.$('#userName');
      if (userNameEl) userNameEl.textContent = this.state.currentUser?.username || 'Você';

      const p = this.state.partner;
      if (p) {
        this.$('#partnerName').textContent = p.username || 'Parceiro';
        this.$('#partnerAvatar').textContent = this.initials(p.username);
        this.updatePartnerSub(p);
      } else {
        this.setNoPartnerUI();
      }

      // Após consumir, limpa backups
      sessionStorage.removeItem('chatTransferData');
      localStorage.removeItem('chatTransferBackup');
    } catch (e) {
      console.error('Erro ao processar dados do chat:', e);
      window.location.href = '/?error=chat_load_failed';
    }
  }

  /* ===================== Socket ===================== */
  setupConnection() {
    this.socket = io({
      reconnection: true,
      reconnectionAttempts: 5,
      reconnectionDelay: 3000,
      withCredentials: true,
      auth: { token: this.getCookie('token') }
    });

    this.socket.on('connect', () => {
      this.state.socketId = this.socket.id;
      this.socket.emit('joinChat', {
        roomId: this.state.roomId,
        sessionId: this.state.sessionId,
        username: this.state.currentUser?.username,
        gender: this.state.currentUser?.gender
      });
      this.socket.emit('chatJoined', {
        roomId: this.state.roomId,
        sessionId: this.state.sessionId
      });
    });

    // Servidor manda dados finais p/ guardar antes do redirect
    this.socket.on('storeChatData', ({ key, data }) => {
      try {
        sessionStorage.setItem(key, JSON.stringify(data));
        localStorage.setItem(key, JSON.stringify(data)); // backup
      } catch (e) {
        console.error('Falha ao salvar chatResultsData', e);
      }
    });

    // Fases
    this.socket.on('setTotalPhases', ({ totalPhases }) => {
      this.$('#totalPhases').textContent = String(totalPhases || 1);
    });

    this.socket.on('phaseTransition', () => {
      this.addSystemMessage('Preparando próximo parceiro...');
    });

    this.socket.on('phaseStarted', (data) => this.handlePhaseStarted(data));
    this.socket.on('clearChatHistory', () => this.clearChat());
    this.socket.on('timeUpdate', ({ timeLeft }) => this.$('#timeLeft').textContent = String(timeLeft));

    // Mensagens
    this.socket.on('newMessage', (data) => this.handleNewMessage(data));
    this.socket.on('messageAck', ({ message, timestamp }) => {
      // eco para quem enviou com timestamp consistente
      this.addMessage({
        text: message,
        type: 'sent',
        name: 'Você',
        age: null,
        ts: timestamp
      });
    });

    // Estados de conexão do parceiro
    this.socket.on('partnerDisconnected', ({ message }) => {
      this.state.partnerOnline = false;
      this.addSystemMessage(message || 'Seu parceiro desconectou.');
      this.showBanner('O usuário não está mais conectado, aguarde a finalização da sala.');
      this.disableComposer(true);
      this.updatePartnerSub(this.state.currentPartner, /*disconnected*/ true);
    });

    this.socket.on('partnerReconnected', () => {
      this.state.partnerOnline = true;
      this.addSystemMessage('Seu parceiro reconectou.');
      this.hideBanner();
      this.disableComposer(false);
      this.updatePartnerSub(this.state.currentPartner, /*disconnected*/ false);
    });

    this.socket.on('userLeftPermanently', () => {
      this.state.partnerOnline = false;
      this.addSystemMessage('Seu parceiro saiu da sala. Aguardando a finalização da fase.');
      this.showBanner('O usuário não está mais conectado, aguarde a finalização da sala.');
      this.disableComposer(true);
      this.updatePartnerSub(this.state.currentPartner, true);
    });

    // Fim de chat -> redireciona (dados já salvos)
    this.socket.on('chatEnded', () => {
      this.addSystemMessage('Chat encerrado. Redirecionando...');
      setTimeout(() => { window.location.href = '/results'; }, 900);
    });
  }

  /* ===================== UI Helpers ===================== */
  initials(name) {
    const n = (name || 'U U').trim().split(/\s+/);
    return ((n[0]?.[0] || 'U') + (n[1]?.[0] || '')).toUpperCase();
  }

  formatTime(tsIso) {
    try {
      const d = tsIso ? new Date(tsIso) : new Date();
      const dd = String(d.getDate()).padStart(2, '0');
      const mm = String(d.getMonth() + 1).padStart(2, '0');
      const hh = String(d.getHours()).padStart(2, '0');
      const mi = String(d.getMinutes()).padStart(2, '0');
      return `${dd}/${mm} ${hh}:${mi}`;
    } catch {
      return '';
    }
  }

  setNoPartnerUI() {
    this.$('#partnerName').textContent = 'Aguardando parceiro...';
    this.$('#partnerAvatar').textContent = '—';
    this.$('#partnerSub').textContent = 'Sem parceiro nesta fase';
    this.showBanner('Sem parceiro nesta fase. Aguarde a próxima rotação.');
    this.disableComposer(true);
  }

  updatePartnerSub(partner, disconnected = false) {
    const ageTxt = (partner && Number.isFinite(partner.age)) ? `${partner.age} anos` : 'idade não informada';
    const status = disconnected ? 'offline' : 'online';
    this.$('#partnerSub').textContent = `${ageTxt} • ${status}`;
  }

  showBanner(text) {
    const b = this.$('#systemBanner');
    const t = this.$('#systemBannerText');
    if (t) t.textContent = text;
    if (b) b.hidden = false;
  }
  hideBanner() {
    const b = this.$('#systemBanner');
    if (b) b.hidden = true;
  }
  disableComposer(disabled) {
    this.$('#messageInput').disabled = !!disabled;
    this.$('#sendButton').disabled = !!disabled;
  }

  clearChat() {
    const box = this.$('#chatContainer');
    box.innerHTML = '';
    this.hideBanner();
    this.disableComposer(false);
    this.addSystemMessage('Nova fase iniciada.');
  }

  addSystemMessage(text) {
    const box = this.$('#chatContainer');
    const el = document.createElement('div');
    el.className = 'msg msg--system';
    el.textContent = text;
    box.appendChild(el);
    box.scrollTop = box.scrollHeight;
  }

  addMessage({ text, type, name, age, ts }) {
    const box = this.$('#chatContainer');
    const wrap = document.createElement('div');
    wrap.className = `msg ${type === 'sent' ? 'msg--sent' : 'msg--recv'}`;

    const meta = document.createElement('div');
    meta.className = 'msg__meta';

    if (type === 'received') {
      const nm = document.createElement('span');
      nm.className = 'msg__name';
      nm.textContent = name || 'Parceiro';
      meta.appendChild(nm);

    } else {
      const nm = document.createElement('span');
      nm.className = 'msg__name';
      nm.textContent = 'Você';
      meta.appendChild(nm);
    }

    const time = document.createElement('span');
    time.className = 'msg__time';
    time.textContent = this.formatTime(ts);
    meta.appendChild(time);

    const txt = document.createElement('div');
    txt.className = 'msg__text';
    txt.textContent = text;

    wrap.appendChild(meta);
    wrap.appendChild(txt);
    box.appendChild(wrap);
    box.scrollTop = box.scrollHeight;
  }

  /* ===================== Business ===================== */
  handlePhaseStarted(data) {
    const { phase, totalPhases, partnerInfo } = data || {};
    this.$('#currentPhase').textContent = String(phase || 1);
    this.$('#totalPhases').textContent = String(totalPhases || 1);

    const partner = partnerInfo || null;
    this.state.currentPartner = partner;
    this.state.partner = partner;

    const nameEl = this.$('#partnerName');
    const avEl = this.$('#partnerAvatar');

    if (!partner) {
      nameEl.textContent = 'Aguardando parceiro...';
      avEl.textContent = '—';
      this.$('#partnerSub').textContent = 'Sem parceiro nesta fase';
      this.showBanner('Sem parceiro nesta fase. Aguarde a próxima rotação.');
      this.disableComposer(true);
      return;
    }

    nameEl.textContent = partner.username || 'Parceiro';
    avEl.textContent = this.initials(partner.username);
    this.updatePartnerSub(partner, false);
    this.hideBanner();
    this.disableComposer(false);

    const exists = this.state.pastPartners.some(p => p.sessionId === partner.sessionId);
    if (!exists) this.state.pastPartners.push(partner);
    this.addSystemMessage(`Agora você está conversando com ${partner.username}${Number.isFinite(partner.age) ? ` (${partner.age})` : ''}.`);
  }

  handleNewMessage(data) {
    // Mensagens RECEBIDAS vêm do servidor com timestamp ISO
    const age = Number.isFinite(this.state.currentPartner?.age) ? this.state.currentPartner.age : null;
    this.addMessage({
      text: data.message,
      type: 'received',
      name: data.senderName || (this.state.currentPartner?.username || 'Parceiro'),
      age,
      ts: data.timestamp
    });
  }

  setupEventListeners() {
    this.$('#messageForm').addEventListener('submit', (e) => {
      e.preventDefault();
      this.sendMessage();
    });
  }

  sendMessage() {
    if (this.$('#sendButton').disabled) return;
    const input = this.$('#messageInput');
    const msg = (input.value || '').trim();
    if (!msg) return;

    // Envia; o servidor devolve messageAck com timestamp
    this.socket.emit('sendMessage', {
      roomId: this.state.roomId,
      message: msg,
      senderName: this.state.currentUser?.username || 'Você'
    });

    input.value = '';
    input.focus();
  }

  getCookie(name) {
    const value = `; ${document.cookie}`;
    const parts = value.split(`; ${name}=`);
    if (parts.length === 2) return parts.pop().split(';').shift();
    return null;
  }
}

document.addEventListener('DOMContentLoaded', () => new ChatClient());
