const db = require('./db');
const logger = require('../config/logger');
const production = false;

class User {
    async create({ username, email, password, gender }) {
        const sql = 'INSERT INTO users (username, email, password, gender) VALUES (?, ?, ?, ?)';
        const result = await db.query(sql, [username, email, password, gender]);
        return result.insertId;
    }

    async findByEmail(email) {
        const sql = 'SELECT * FROM users WHERE email = ?';
        const [user] = await db.query(sql, [email]);
        return user;
    }

    async findById(id) {
        const sql = 'SELECT * FROM users WHERE id = ?';
        const [user] = await db.query(sql, [id]);
        //console.log('Pegou findById')
        //console.log(user)
        return user;
    }

    async update(userId, data) {
        const fields = [];
        const values = [];

        const allow = [
            'username', 'gender', 'birthDate', 'height', 'weight', 'profilePicture',
            'pronouns', 'bio', 'occupation', 'city', 'state',
            'relationship_goals', 'relationship_type', 'has_children', 'drinks', 'smokes',
            'political_orientation', 'religion',
            'kitchen_persona', 'diet_style', 'pets', 'coffee', 'sports_role', 'party_style', 'gangster_persona'
        ];

        // mapa campo->coluna
        const colMap = {
            birthDate: 'birth_date',
            profilePicture: 'profile_picture'
        };

        for (const k of allow) {
            // undefined -> ignora; null -> seta NULL; string vazia nunca chega aqui porque normalizamos no service
            if (data[k] === undefined) continue;

            const col = colMap[k] || k; // nome real da coluna
            if (data[k] === null) {
                fields.push(`\`${col}\` = NULL`);
            } else {
                fields.push(`\`${col}\` = ?`);
                values.push(data[k]);
            }
        }

        // JSONs
        if (data.social_links !== undefined) {
            fields.push('`social_links` = ?'); values.push(JSON.stringify(data.social_links));
        }
        if (data.with_you !== undefined) {
            fields.push('`with_you` = ?'); values.push(JSON.stringify(data.with_you));
        }
        if (data.perfect_day !== undefined) {
            fields.push('`perfect_day` = ?'); values.push(JSON.stringify(data.perfect_day));
        }

        if (!fields.length) return false;
        values.push(userId);
        const sql = `UPDATE users SET ${fields.join(', ')} WHERE id = ?`;
        const result = await db.query(sql, values);
        return result.affectedRows > 0;
    }



    async updateImage(userId, data) {
        const sql = `UPDATE users SET profile_picture = ? WHERE id = ?`;
        const result = await db.query(sql, [data.profilePicture, userId]);
        return result.affectedRows > 0;
    }


    async addGalleryImage(userId, imagePath) {
        const query = 'INSERT INTO gallery_images (id_user, image_path) VALUES (?, ?)';
        const result = await db.query(query, [userId, imagePath]);
        return result.insertId; // MySQL retorna o insertId diretamente no result
    }

    async getGalleryImages(userId) {
        //console.log('Galeria user')
        const query = 'SELECT * FROM gallery_images WHERE id_user = ? ORDER BY created_at DESC';
        const rows = await db.query(query, [userId]);
        return rows;
    }

    async deleteGalleryImage(userId, imageId) {
        const querySelect = 'SELECT image_path FROM gallery_images WHERE id = ? AND id_user = ?';
        const rows = await db.query(querySelect, [imageId, userId]);

        if (rows.length === 0) {
            throw new Error('Imagem não encontrada ou não pertence ao usuário');
        }

        const imagePath = rows[0].image_path;
        const queryDelete = 'DELETE FROM gallery_images WHERE id = ? AND id_user = ?';
        await db.query(queryDelete, [imageId, userId]);

        return imagePath;
    }

    // Método adicional para buscar apenas os dados básicos do usuário
    async getBasicInfo(userId) {
        const sql = 'SELECT id, username, email, gender FROM users WHERE id = ?';
        const [user] = await db.query(sql, [userId]);
        return user;
    }

    async updateExplorer(userId, { explorer, explorer_type }) {
        // defesa: normaliza explorer para 0/1 e whitelista explorer_type
        const explorerFlag = explorer ? 1 : 0;
        const type = ['public', 'private'].includes(explorer_type) ? explorer_type : 'public';

        const query = 'UPDATE users SET explorer = ?, explorer_type = ? WHERE id = ?';
        const result = await db.query(query, [explorerFlag, type, userId]);
        return result.affectedRows > 0;
    }

    async getExplorerUsers(currentUserId) {
        if (currentUserId == null) throw new Error('currentUserId inválido');

        const query = `
      SELECT DISTINCT
        u.id,
        u.username,
        u.profile_picture AS profilePicture,
        u.explorer_type
      FROM users u
      LEFT JOIN friendships f 
        ON (
          (f.user1_id = u.id AND f.user2_id = ?) OR
          (f.user2_id = u.id AND f.user1_id = ?)
        )
      WHERE 
        u.explorer = 1
        AND u.id <> ?
        AND NOT EXISTS (
          SELECT 1 FROM blacklists b
          WHERE 
            (b.blocker_id = u.id AND b.blocked_id = ?)
            OR
            (b.blocker_id = ? AND b.blocked_id = u.id)
        )
        AND (
          u.explorer_type = 'public'
          OR (u.explorer_type = 'private' AND f.id IS NOT NULL)
        )
      ORDER BY u.id DESC
    `;

        // ⚠️ sem desestruturar — db.query já retorna o array de linhas
        const rows = await db.query(query, [
            currentUserId, currentUserId, currentUserId, currentUserId, currentUserId
        ]);

        // sempre devolve array
        return rows || [];
    }

    async listInterests() {
        const q = 'SELECT id, slug, label, category FROM interests ORDER BY category, label';
        return await db.query(q);
    }
    async getUserInterests(userId) {
        const q = `
      SELECT i.slug, i.label
      FROM user_interests ui
      JOIN interests i ON i.id = ui.interest_id
      WHERE ui.user_id = ?
      ORDER BY i.label`;
        return await db.query(q, [userId]);
    }
    async setUserInterests(userId, slugs = []) {
        // resolve slugs -> ids
        if (!Array.isArray(slugs)) slugs = [];
        const inClause = slugs.length ? slugs.map(() => '?').join(',') : null;
        let ids = [];
        if (inClause) {
            const q = `SELECT id, slug FROM interests WHERE slug IN (${inClause})`;
            const rows = await db.query(q, slugs);
            ids = rows.map(r => r.id);
        }
        await db.query('DELETE FROM user_interests WHERE user_id = ?', [userId]);
        if (ids.length) {
            const values = ids.map(id => `(${userId}, ${id})`).join(',');
            await db.query(`INSERT INTO user_interests (user_id, interest_id) VALUES ${values}`);
        }
        return true;
    }

    /**
 * Retorna true se A bloqueou B OU B bloqueou A.
 * Usa a tabela `blacklists(blocker_id, blocked_id)`.
 */
    async isBlockedBetween(userIdA, userIdB) {
        if (userIdA == null || userIdB == null) return false;
        const sql = `
            SELECT 1
            FROM blacklists
            WHERE (blocker_id = ? AND blocked_id = ?)
               OR (blocker_id = ? AND blocked_id = ?)
            LIMIT 1
        `;
        const rows = await db.query(sql, [userIdA, userIdB, userIdB, userIdA]);
        return Array.isArray(rows) && rows.length > 0;
    }

    /**
     * Lista IDs que o user bloqueou (direção A->B).
     * Útil para filtros de feed / explore.
     */
    async blockedIdsBy(userId) {
        const sql = `SELECT blocked_id FROM blacklists WHERE blocker_id = ?`;
        const rows = await db.query(sql, [userId]);
        return (rows || []).map(r => r.blocked_id);
    }

    /**
     * Lista IDs que bloquearam o user (direção B->A).
     */
    async blockerIdsOf(userId) {
        const sql = `SELECT blocker_id FROM blacklists WHERE blocked_id = ?`;
        const rows = await db.query(sql, [userId]);
        return (rows || []).map(r => r.blocker_id);
    }




}

module.exports = new User();