// C:\src\Clientes\Datetime\middlewares\authMiddleware.js
const jwt = require('jsonwebtoken');
const User = require('../models/users');
const { AppError } = require('../config/errors');

const jwtSecret = process.env.JWT_SECRET;
if (!jwtSecret) {
  throw new Error('JWT_SECRET não configurado');
}

/** Extrai access token:
 * - requisições de página/SPA: cookie "token"
 * - APIs: Authorization: Bearer <jwt> (fallback) ou cookie "token"
 */
function getAccessToken(req) {
  const cookieToken = req.cookies?.token;
  const auth = req.get('Authorization') || '';
  const isBearer = auth.toLowerCase().startsWith('bearer ');
  const headerToken = isBearer ? auth.slice(7).trim() : null;
  return cookieToken || headerToken || null;
}

async function verifyUserToken(token) {
  if (!token) throw new AppError('Token não fornecido', 401);

  const decoded = jwt.verify(token, jwtSecret);
  const userId = decoded.sub || decoded.id; // compatível com as duas versões
  if (!userId) throw new AppError('Token inválido (sem sub/id)', 401);

  const user = await User.findById(userId);
  if (!user) throw new AppError('Usuário não encontrado', 404);

  return user;
}

/** Limpa o cookie de access token com os mesmos atributos do set */
function clearAccessCookie(res) {
  res.clearCookie('token', {
    httpOnly: true,
    secure: process.env.NODE_ENV === 'production',
    sameSite: 'lax',
    path: '/'
  });
}

/* ===========================
 * Páginas HTML (redirect)
 * =========================== */
exports.requireAuth = async (req, res, next) => {
  try {
    const token = getAccessToken(req);
    const user = await verifyUserToken(token);
    req.user = user;
    return next();
  } catch (error) {
    clearAccessCookie(res);

    // Expirado → mande o usuário para login com flag; o front pode chamar /auth/refresh
    if (error.name === 'TokenExpiredError') {
      return res.redirect('/auth/login?expired=1');
    }
    // Assinatura inválida / sem token / usuário não encontrado
    return res.redirect('/auth/login');
  }
};

/* ===========================
 * APIs JSON (não redireciona)
 * - Retorna 401 com code padronizado:
 *   TOKEN_EXPIRED | UNAUTHORIZED | USER_NOT_FOUND
 * =========================== */
exports.apiAuth = async (req, res, next) => {
  try {
    const token = getAccessToken(req);
    const user = await verifyUserToken(token);
    req.user = user;
    return next();
  } catch (error) {
    clearAccessCookie(res);

    if (error.name === 'TokenExpiredError') {
      return res.status(401).json({
        ok: false,
        code: 'TOKEN_EXPIRED',
        message: 'Access token expirado. Renove o token com /auth/refresh.'
      });
    }

    if (error.message === 'Usuário não encontrado') {
      return res.status(401).json({
        ok: false,
        code: 'USER_NOT_FOUND',
        message: 'Usuário não encontrado.'
      });
    }

    return res.status(401).json({
      ok: false,
      code: 'UNAUTHORIZED',
      message: 'Não autorizado. Faça login novamente.'
    });
  }
};
