// controllers/lobbyController.js
require('dotenv').config();
const jwt = require('jsonwebtoken');
const jwtSecret = process.env.JWT_SECRET || 'sua_chave_secreta';
const lobbyService = require('../services/lobbyService');
const User = require('../models/users'); // <- buscar do DB para obter gênero correto

// mesma normalização usada no service/server
function normalizeGender(g) {
  if (!g) return 'other';
  const s = String(g).trim().toLowerCase();
  const maleSet = new Set(['male','m','masculino','homem','masc']);
  const femaleSet = new Set(['female','f','feminino','mulher','fem']);
  if (maleSet.has(s)) return 'male';
  if (femaleSet.has(s)) return 'female';
  return 'other';
}

class LobbyController {
  constructor() {
    this.io = null;
    this.rulesUrl = process.env.LOBBY_RULES_URL || null;
  }

  initialize(ioInstance) {
    this.io = ioInstance;
    this.setupSocketHandlers();
  }

  setupSocketHandlers() {
    this.io.on('connection', socket => {
      socket.on('joinLobby', async (data, callback) => {
        try {
          const user = socket.user;
          if (!user) throw new Error('Usuário não autenticado');

          const { roomId, sessionId } = data;
          if (!roomId || !sessionId) throw new Error('Dados incompletos');

          // socket.user.gender já vem normalizado do middleware (server.js)
          const joinResult = await lobbyService.joinRoom(
            {
              userId: user.id,
              username: user.username,
              gender: user.gender,
              sessionId
            },
            roomId,
            socket.id
          );

          socket.join(roomId);

          callback?.({
            status: 'success',
            ...joinResult.room,
            currentUsers: joinResult.room.users.size
          });

          await this.emitRoomUpdate(roomId);

          if (joinResult.triggerCountdown) {
            await this.startCountdown(roomId);
          }
        } catch (err) {
          callback?.({ status: 'error', message: err.message });
        }
      });

      socket.on('disconnect', () => {
        const sock = this.io.sockets.sockets.get(socket.id);
        if (sock?.inTransition) return;
        this.handleDisconnect(socket.id);
      });

      socket.on('startCountdown', async ({ roomId }) => {
        await this.startCountdown(roomId);
      });

      socket.on('chatJoined', ({ sessionId, roomId }) => {
        const roomCleaned = lobbyService.handleChatJoined(sessionId, roomId);
        if (roomCleaned) {
          this.emitRoomUpdate(roomId);
        }
      });

      socket.on('leaveLobby', ({ roomId, sessionId }) => {
        if (socket.inTransition) return;
        this.handleLeaveLobby(socket.id, roomId, sessionId);
      });

      socket.on('chatFinished', ({ roomId }) => {
        this.emitRoomUpdate(roomId);
      });
    });
  }

  async startCountdown(roomId) {
    await lobbyService.startCountdown(roomId, (eventType, targetId, data) => {
      if (eventType === 'countdownUpdate') {
        this.io.to(targetId).emit('countdownUpdate', data);
      } else if (eventType === 'countdownCancelled') {
        this.io.to(targetId).emit('countdownCancelled');
        this.emitRoomUpdate(targetId); // reemite estado do lobby
      } else if (eventType === 'redirectToChat') {
        const socket = this.io.sockets.sockets.get(targetId);
        if (socket) {
          socket.inTransition = true;
          socket.emit('redirectToChat', data);
        }
      }
    });
  }

  handleLeaveLobby(socketId, roomId, sessionId) {
    const result = lobbyService.leaveRoom(socketId, roomId, sessionId);
    if (result.roomUpdated) {
      if (result.countdownCancelled) {
        this.io.to(roomId).emit('countdownCancelled');
      }
      this.emitRoomUpdate(roomId);
    }
  }

  handleDisconnect(socketId) {
    const socket = this.io.sockets.sockets.get(socketId);
    if (socket?.inTransition) return;

    const disconnectResult = lobbyService.handleUserDisconnect(socketId, (eventType, targetId) => {
      if (eventType === 'countdownCancelled') {
        this.io.to(targetId).emit('countdownCancelled');
      }
    });

    if (disconnectResult.roomChanged) {
      this.io.emit('updateRooms', lobbyService.getPublicRooms());
    }
  }

  async emitRoomUpdate(roomId) {
    const room = lobbyService.rooms.get(roomId);
    if (!room) return;

    const blocking = await lobbyService.getBlockingStatus(roomId);

    this.io.to(roomId).emit('lobbyUpdate', {
      currentUsers: room.users.size,
      maxUsers: room.maxUsers,
      userData: lobbyService.getRoomUsers(roomId),
      hasBlockingConflict: !!blocking.hasBlockingConflict,
      rulesUrl: this.rulesUrl
    });

    this.io.emit('updateRooms', lobbyService.getPublicRooms());
  }

  /**
   * HTTP /lobby/enter:
   * - Valida token
   * - Busca usuário no DB (para obter gênero real)
   * - Normaliza gênero
   * - Valida regra de capacidade por gênero
   * - Reserva entrada (sem socketId)
   */
  async handleHttpEnter(req, res) {
    try {
      const { roomId } = req.query;
      const token = req.cookies.token;
      if (!token) throw new Error('Não autenticado');

      const decoded = jwt.verify(token, jwtSecret);
      const userId = decoded.sub || decoded.id;
      if (!userId) throw new Error('Token inválido');

      const room = lobbyService.rooms.get(roomId);
      if (!room) throw new Error('Sala não encontrada');

      // ⚠️ Busca o usuário no DB para pegar gênero atualizado/confiável
      const dbUser = await User.findById(Number(userId));
      if (!dbUser) throw new Error('Usuário não encontrado');

      const gender = normalizeGender(dbUser.gender || decoded.gender || null);

      // Valida capacidade com gênero NORMALIZADO
      if (!lobbyService.canUserJoinRoom(room, gender)) {
        throw new Error('Sala cheia ou sem vaga para o seu gênero');
      }

      const sessionId = lobbyService.generateSessionId();

      // Reserva a vaga no estado do lobby (sem socketId aqui)
      await lobbyService.joinRoom(
        {
          userId: Number(userId),
          username: dbUser.username || decoded.username || null,
          gender,
          sessionId
        },
        roomId,
        null
      );

      res.json({ success: true, roomId, sessionId });
    } catch (err) {
      res.status(400).json({ success: false, message: err.message });
    }
  }
}

const controller = new LobbyController();
module.exports = controller;
module.exports.getRooms = (req, res) => res.json(lobbyService.getPublicRooms());
module.exports.enterLobby = (req, res) => controller.handleHttpEnter(req, res);
