const friendshipService = require('../services/friendshipService');
const { BadRequestError, NotFoundError, ForbiddenError, ConflictError } = require('../config/errors');

function parseFriendId(friendId) {
  if (!isNaN(friendId)) return parseInt(friendId, 10);
  if (typeof friendId === 'string' && friendId.includes('_')) {
    const n = friendId.split('_')[1];
    if (!isNaN(n)) return parseInt(n,10);
  }
  throw new BadRequestError('Formato de ID inválido. Use número ou "user_X".');
}

function handleError(res, error) {
  console.error('Erro no controller:', error);
  if (error instanceof BadRequestError) return res.status(400).json({ error: error.message });
  if (error instanceof NotFoundError) return res.status(404).json({ error: error.message });
  if (error instanceof ForbiddenError) return res.status(403).json({ error: error.message });
  if (error instanceof ConflictError) return res.status(409).json({ error: error.message });
  return res.status(500).json({ error: error.message || 'Erro interno no servidor' });
}

module.exports = {
  // Requests
  sendFriendRequest: async (req, res) => {
    try {
      const friendId = parseFriendId(req.body.friendId);
      const result = await friendshipService.sendFriendRequest(req.user.id, friendId);
      res.status(200).json(result);
    } catch (e) { handleError(res, e); }
  },
  respondToFriendRequest: async (req, res) => {
    try {
      const { action } = req.body; // 'accept' | 'reject' | 'cancel'
      const result = await friendshipService.respondToRequest(req.user.id, req.params.requestId, action);
      res.status(200).json(result);
    } catch (e) { handleError(res, e); }
  },
  listFriendRequests: async (req, res) => {
    try {
      const r = await friendshipService.listRequests(req.user.id);
      res.status(200).json(r);
    } catch (e) { handleError(res, e); }
  },
  countFriendRequests: async (req, res) => {
    try {
      const c = await friendshipService.countRequests(req.user.id);
      res.status(200).json({ count:c });
    } catch (e) { handleError(res, e); }
  },

  // Friends
  listFriends: async (req, res) => {
    try {
      const r = await friendshipService.listFriends(req.user.id, req.query.search);
      res.status(200).json(r);
    } catch (e) { handleError(res, e); }
  },
  searchFriends: async (req, res) => {
    try {
      const r = await friendshipService.searchFriends(req.user.id, req.query.term);
      res.status(200).json(r);
    } catch (e) { handleError(res, e); }
  },
  removeFriend: async (req, res) => {
    try {
      const r = await friendshipService.removeFriend(req.user.id, req.params.friendId);
      res.status(200).json(r);
    } catch (e) { handleError(res, e); }
  },

  // Status + Views
  checkFriendshipStatus: async (req, res) => {
    try {
      const friendId = parseFriendId(req.params.userId);
      const r = await friendshipService.checkStatus(req.user.id, friendId);
      res.status(200).json(r);
    } catch (e) { handleError(res, e); }
  },
  viewExplorerProfile: async (req, res) => {
    try {
      const profile = await friendshipService.viewExplorerProfile(req.user.id, req.params.friendId);
      res.status(200).json(profile);
    } catch (e) { handleError(res, e); }
  },
  viewFriendProfile: async (req, res) => {
    try {
      const profile = await friendshipService.viewProfile(req.user.id, req.params.friendId, true);
      res.status(200).json(profile);
    } catch (e) { handleError(res, e); }
  },
  getGaleria: async (req, res) => {
    try {
      const gal = await friendshipService.getGaleria(req.params.friendId, req.user.id);
      res.status(200).json(gal);
    } catch (e) { handleError(res, e); }
  },

  // Blocks
  blockUser: async (req, res) => {
    try {
      const r = await friendshipService.blockUser(req.user.id, req.body.targetId);
      res.status(200).json(r);
    } catch (e) { handleError(res, e); }
  },
  unblockUser: async (req, res) => {
    try {
      const r = await friendshipService.unblockUser(req.user.id, req.params.userId);
      res.status(200).json({ message: r.message });
    } catch (e) { handleError(res, e); }
  },
  listBlockedUsers: async (req, res) => {
    try {
      const r = await friendshipService.listBlockedUsers(req.user.id);
      res.status(200).json(r);
    } catch (e) { handleError(res, e); }
  },
  checkIfBlocked: async (req, res) => {
    try {
      const isBlocked = await friendshipService.checkIfBlocked(req.user.id, req.params.userId);
      res.json({ isBlocked });
    } catch (e) { handleError(res, e); }
  }
};
