const db = require('../models/db');

/**
 * Calcula idade simples a partir de YYYY-MM-DD (ou Date).
 */
function calcAge(birth) {
  if (!birth) return null;
  const d = new Date(birth);
  if (isNaN(d)) return null;
  const now = new Date();
  let age = now.getFullYear() - d.getFullYear();
  const m = now.getMonth() - d.getMonth();
  if (m < 0 || (m === 0 && now.getDate() < d.getDate())) age--;
  return age >= 0 ? age : null;
}

const chatHistoryController = {
  getHistory: async (req, res) => {
    try {
      const userId = req.params.id;

      // Puxa histórico e a data de nascimento do parceiro para calcular idade
      const rows = await db.query(
        `
        SELECT
          ch.current_user_id,
          ch.current_user_username,
          ch.partner_id,
          ch.partner_username,
          ch.chat_end_time,          -- usamos o fim como "momento da conversa" (quando foi registrada)
          u.birth_date AS partner_birth_date
        FROM chat_history ch
        LEFT JOIN users u ON u.id = ch.partner_id
        WHERE ch.current_user_id = ?
        ORDER BY ch.chat_end_time DESC
        `,
        [userId]
      );

      if (!rows || rows.length === 0) {
        return res.status(200).json({ history: [] });
      }

      // Calcula idade no servidor para já enviar pronto
      const history = rows.map(r => ({
        current_user_id: r.current_user_id,
        current_user_username: r.current_user_username,
        partner_id: r.partner_id,
        partner_username: r.partner_username,
        chat_end_time: r.chat_end_time,
        partner_age: calcAge(r.partner_birth_date)
      }));

      return res.status(200).json({ history });
    } catch (error) {
      console.error('Erro ao buscar histórico:', error);
      res.status(500).json({ error: 'Erro interno do servidor' });
    }
  },
};

module.exports = chatHistoryController;
