// controllers/chatController.js
const db = require('../models/db');
const Users = require('../models/users');
const EventEmitter = require('events');
const { phaseDurationSec } = require('../config/appConfig');

function calcAgeFromBirthDate(birthDate) {
  if (!birthDate) return null;
  try {
    const d = (birthDate instanceof Date) ? birthDate : new Date(birthDate);
    if (isNaN(d)) return null;
    const now = new Date();
    let age = now.getFullYear() - d.getFullYear();
    const m = now.getMonth() - d.getMonth();
    if (m < 0 || (m === 0 && now.getDate() < d.getDate())) age--;
    return age >= 0 ? age : null;
  } catch { return null; }
}

// Helper: extrai o número do userId (aceita 'user_123' ou '123')
function toNumericId(maybePrefixed) {
  if (!maybePrefixed) return null;
  const s = String(maybePrefixed);
  const m = s.match(/(\d+)$/);
  return m ? Number(m[1]) : Number(s);
}

class ChatController extends EventEmitter {
  constructor() {
    super();
    this.io = null;
    this.chatRooms = new Map();
  }

  initialize(ioInstance) {
    this.io = ioInstance;
    this.setupSocketHandlers();
  }

  setupSocketHandlers() {
    this.io.on('connection', socket => {
      socket.on('error', (error) => {
        console.error(`Erro no socket ${socket.id}:`, error);
      });

      socket.on('joinChat', async ({ roomId, sessionId, username, gender }, callback) => {
        try {
          if (!roomId || !sessionId) throw new Error('Dados incompletos');

          // Cria/obtém sala
          if (!this.chatRooms.has(roomId)) {
            this.chatRooms.set(roomId, {
              participants: new Map(),
              phases: [],
              currentPhase: 0,
              timer: null,
              pairs: new Map(),
              creationTime: Date.now(),
              category: this.getCategoryFromRoomId(roomId),
              phaseDurationSec,
              phaseEndsAt: null,
              graceReconnectMs: 15000
            });
          }
          const room = this.chatRooms.get(roomId);

          // Enriquecer com dados do DB (idade)
          let birthDate = null;
          let city = null;
          let state = null;
          try {
            if (socket.user?.id) {
              const u = await Users.findById(Number(socket.user.id));
              birthDate = u?.birth_date || null;
              city = u?.city || null;
              state = u?.state || u?.uf || null; // tolerante a nome da coluna
            }
          } catch { /* silencioso */ }

          // Registrar/atualizar participante
          if (room.participants.has(sessionId)) {
            const participant = room.participants.get(sessionId);
            participant.socket = socket;
            participant.socketId = socket.id;
            participant.userInfo.username = username ?? participant.userInfo.username ?? 'Anônimo';
            participant.userInfo.gender = gender ?? participant.userInfo.gender ?? 'unknown';
            participant.userInfo.birthDate = participant.userInfo.birthDate ?? birthDate ?? null;
            participant.userInfo.city = participant.userInfo.city ?? city ?? null;
            participant.userInfo.state = participant.userInfo.state ?? state ?? null;
          } else {
            room.participants.set(sessionId, {
              socket,
              socketId: socket.id,
              sessionId,
              userInfo: {
                username: username || socket.user?.username || 'Anônimo',
                gender: gender || socket.user?.gender || 'unknown',
                birthDate: birthDate || null,
                city: city || null,
                state: state || null,
                userId: socket.user?.id ? `user_${socket.user.id}` : null
              },
              joinTime: Date.now()
            });
          }

          socket.join(roomId);

          // Notifica parceiro sobre reconexão (se já havia pareamento ativo)
          const partnerId = room.pairs.get(socket.id);
          if (partnerId) {
            const partnerSocket = this.io.sockets.sockets.get(partnerId);
            if (partnerSocket) partnerSocket.emit('partnerReconnected');
          }

          this.checkRoomReady(roomId);
          callback?.({ status: 'success', roomId });
        } catch (error) {
          console.error('Erro ao entrar no chat:', error);
          callback?.({ status: 'error', message: error.message });
          if (String(error.message).includes('não encontrada')) {
            setTimeout(() => { socket.emit('redirectToLobby'); }, 1000);
          }
        }
      });

      socket.on('sendMessage', ({ roomId, message, senderName }, callback) => {
        try {
          if (!roomId || !message) throw new Error('Dados incompletos');
          this.handleMessage(socket, roomId, message, senderName);
          callback?.({ status: 'success' });
        } catch (error) {
          console.error('Erro ao enviar mensagem:', error);
          callback?.({ status: 'error', message: error.message });
        }
      });

      socket.on('disconnect', () => this.handleDisconnect(socket.id));
    });
  }

  getCategoryFromRoomId(roomId) {
    if (roomId.includes('heterossexual')) return 'heterossexual';
    if (roomId.includes('homossexual')) return 'homossexual';
    if (roomId.includes('lgbt')) return 'lgbt';
    return 'unknown';
  }

  checkRoomReady(roomId) {
    const room = this.chatRooms.get(roomId);
    if (!room) return false;

    const participants = Array.from(room.participants.values());
    const allReady = participants.every(p => p.userInfo.username && p.userInfo.gender);
    const requiredUsers = this.getMaxUsersForRoom(roomId);

    if (participants.length >= requiredUsers && allReady) {
      this.setupChatRoom(roomId);
      this.startNextPhase(roomId);
      return true;
    }
    return false;
  }

  getMaxUsersForRoom(roomId) {
    if (roomId.includes('_sala1')) return 2;
    if (roomId.includes('_sala2')) return 4;
    if (roomId.includes('_sala3')) return 6;
    return 2;
  }

  setupChatRoom(roomId) {
    const room = this.chatRooms.get(roomId);
    const participants = Array.from(room.participants.values());

    if (room.category === 'heterossexual') {
      const men = participants.filter(p => p.userInfo.gender === 'male');
      const women = participants.filter(p => p.userInfo.gender === 'female');
      if (men.length !== women.length) return; // espera balancear
      this.setupHeterosexualPhases(roomId, men, women);
    } else if (room.category === 'homossexual') {
      const firstGender = participants[0]?.userInfo.gender;
      if (!participants.every(p => p.userInfo.gender === firstGender)) return;
      this.setupHomosexualPhases(roomId, participants);
    } else {
      this.setupGenericPhases(roomId, participants);
    }
  }

  setupHeterosexualPhases(roomId, men, women) {
    const room = this.chatRooms.get(roomId);
    const n = men.length;
    room.phases = [];
    if (n === 1) {
      room.phases.push({ pairs: [[men[0], women[0]]] });
    } else if (n === 2) {
      room.phases.push({ pairs: [[men[0], women[0]], [men[1], women[1]]] });
      room.phases.push({ pairs: [[men[0], women[1]], [men[1], women[0]]] });
    } else if (n === 3) {
      room.phases.push({ pairs: [[men[0], women[0]], [men[1], women[1]], [men[2], women[2]]] });
      room.phases.push({ pairs: [[men[0], women[1]], [men[1], women[2]], [men[2], women[0]]] });
      room.phases.push({ pairs: [[men[0], women[2]], [men[1], women[0]], [men[2], women[1]]] });
    }
    room.totalPhases = room.phases.length;
    this.io.to(roomId).emit('setTotalPhases', { totalPhases: room.totalPhases });
  }

  setupHomosexualPhases(roomId, participants) {
    const room = this.chatRooms.get(roomId);
    const n = participants.length;
    room.phases = [];
    if (n === 2) {
      room.phases.push({ pairs: [[participants[0], participants[1]]] });
    } else if (n === 4) {
      room.phases.push({ pairs: [[participants[0], participants[1]], [participants[2], participants[3]]] });
      room.phases.push({ pairs: [[participants[0], participants[2]], [participants[1], participants[3]]] });
    } else if (n === 6) {
      room.phases.push({ pairs: [[participants[0], participants[1]], [participants[2], participants[3]], [participants[4], participants[5]]] });
      room.phases.push({ pairs: [[participants[0], participants[2]], [participants[1], participants[4]], [participants[3], participants[5]]] });
      room.phases.push({ pairs: [[participants[0], participants[3]], [participants[1], participants[5]], [participants[2], participants[4]]] });
    }
    room.totalPhases = room.phases.length;
    this.io.to(roomId).emit('setTotalPhases', { totalPhases: room.totalPhases });
  }

  setupGenericPhases(roomId, participants) {
    const room = this.chatRooms.get(roomId);
    const n = participants.length;
    room.phases = [];
    if (n === 2) {
      room.phases.push({ pairs: [[participants[0], participants[1]]] });
    } else if (n === 4) {
      room.phases.push({ pairs: [[participants[0], participants[1]], [participants[2], participants[3]]] });
      room.phases.push({ pairs: [[participants[0], participants[2]], [participants[1], participants[3]]] });
    } else if (n === 6) {
      room.phases.push({ pairs: [[participants[0], participants[1]], [participants[2], participants[3]], [participants[4], participants[5]]] });
      room.phases.push({ pairs: [[participants[0], participants[2]], [participants[1], participants[4]], [participants[3], participants[5]]] });
      room.phases.push({ pairs: [[participants[0], participants[3]], [participants[1], participants[5]], [participants[2], participants[4]]] });
    }
    room.totalPhases = room.phases.length;
    this.io.to(roomId).emit('setTotalPhases', { totalPhases: room.totalPhases });
  }

  startNextPhase(roomId) {
    const room = this.chatRooms.get(roomId);
    if (!room) return;

    if (room.timer) clearInterval(room.timer);
    if (room.currentPhase >= room.phases.length) return this.endChat(roomId);

    this.io.to(roomId).emit('clearChatHistory');
    this.io.to(roomId).emit('phaseTransition', { message: 'Preparando próximo parceiro...' });

    setTimeout(async () => {
      const currentPhase = room.phases[room.currentPhase];
      room.pairs.clear();

      const everyone = Array.from(room.participants.values());
      const blockedSet = new Set();

      // Monta mapa com checagem de bloqueio
      for (const [u1, u2] of currentPhase.pairs) {
        if (!u1 || !u2) continue;
        if (!u1.socketId || !u2.socketId) continue;

        const aId = toNumericId(u1.userInfo.userId || u1.userId);
        const bId = toNumericId(u2.userInfo.userId || u2.userId);

        // eslint-disable-next-line no-await-in-loop
        const blocked = await Users.isBlockedBetween(aId, bId);
        if (blocked) {
          blockedSet.add(u1.socketId);
          blockedSet.add(u2.socketId);
          continue; // não pareia
        }

        room.pairs.set(u1.socketId, u2.socketId);
        room.pairs.set(u2.socketId, u1.socketId);
      }

      // Emite phaseStarted para TODOS (bloqueados/sem par recebem null + sinalizador)
      everyone.forEach(p => {
        const partnerSocketId = room.pairs.get(p.socketId);

        if (blockedSet.has(p.socketId) || !partnerSocketId) {
          p.socket?.emit('phaseStarted', {
            phase: room.currentPhase + 1,
            totalPhases: room.totalPhases,
            partnerInfo: null,
            phaseDurationSec: room.phaseDurationSec,
            blocked: blockedSet.has(p.socketId),
            reason: blockedSet.has(p.socketId)
              ? 'Seu parceiro está bloqueado. Aguarde até o fim desta fase.'
              : 'Sem parceiro nesta fase. Aguarde a próxima rotação.'
          });
          return;
        }

        const partner = everyone.find(x => x.socketId === partnerSocketId);
        if (!partner) {
          p.socket?.emit('phaseStarted', {
            phase: room.currentPhase + 1,
            totalPhases: room.totalPhases,
            partnerInfo: null,
            phaseDurationSec: room.phaseDurationSec,
            reason: 'Sem parceiro nesta fase. Aguarde a próxima rotação.'
          });
          return;
        }

        const partnerInfo = {
          username: partner.userInfo.username,
          // NÃO enviar pronome/gênero pro front de results
          sessionId: partner.sessionId,
          socketId: partner.socketId,
          userId: partner.userInfo.userId,
          age: calcAgeFromBirthDate(partner.userInfo.birthDate),
          city: partner.userInfo.city || null,
          state: partner.userInfo.state || null
        };

        p.socket?.emit('phaseStarted', {
          phase: room.currentPhase + 1,
          totalPhases: room.totalPhases,
          partnerInfo,
          phaseDurationSec: room.phaseDurationSec
        });
      });

      room.currentPhase++;

      // Timer baseado no ENV + tolerante a suspensões
      let timeLeft = room.phaseDurationSec;
      room.phaseEndsAt = Date.now() + (timeLeft * 1000);
      this.io.to(roomId).emit('timeUpdate', { timeLeft });

      room.timer = setInterval(() => {
        const remaining = Math.max(0, Math.ceil((room.phaseEndsAt - Date.now()) / 1000));
        this.io.to(roomId).emit('timeUpdate', { timeLeft: remaining });
        if (remaining <= 0) {
          clearInterval(room.timer);
          room.timer = null;
          room.phaseEndsAt = null;
          this.startNextPhase(roomId);
        }
      }, 1000);
    }, 1200);
  }

  async endChat(roomId) {
    const room = this.chatRooms.get(roomId);
    if (!room) return;

    const endTime = new Date();

    for (const participant of room.participants.values()) {
      const partnersMap = new Map(); // evita duplicatas por sessionId

      for (const phase of room.phases) {
        for (const pair of phase.pairs) {
          if (!Array.isArray(pair)) continue;
          if (pair.some(p => p?.sessionId === participant.sessionId)) {
            pair.forEach(p => {
              if (p?.sessionId !== participant.sessionId) {
                const age = calcAgeFromBirthDate(p.userInfo?.birthDate);
                partnersMap.set(p.sessionId, {
                  userId: p.userInfo?.userId,
                  username: p.userInfo?.username,
                  sessionId: p.sessionId,
                  age: Number.isFinite(age) ? age : null,
                  city: p.userInfo?.city || null,
                  state: p.userInfo?.state || null
                });
              }
            });
          }
        }
      }

      const partners = Array.from(partnersMap.values());

      const resultData = {
        currentUser: {
          userId: participant.userInfo.userId,
          username: participant.userInfo.username,
          gender: participant.userInfo.gender,
          sessionId: participant.sessionId
        },
        partners,
        roomId,
        endTime: endTime.toISOString()
      };

      try {
        for (const partner of partners) {
          await db.query(
            'INSERT INTO chat_history (current_user_id, current_user_username, partner_id, partner_username, chat_end_time) VALUES (?, ?, ?, ?, ?)',
            [
              String(resultData.currentUser.userId || '').replace(/^user_/, ''),
              resultData.currentUser.username,
              String(partner.userId || '').replace(/^user_/, ''),
              partner.username,
              endTime
            ]
          );
        }
      } catch (error) {
        console.error('Erro ao salvar histórico de chat:', error);
      }

      // Primeiro, garanta que o cliente salvou em sessionStorage/localStorage
      participant.socket?.emit('storeChatData', {
        key: 'chatResultsData',
        data: resultData
      });

      // Depois, sinalize o redirecionamento
      participant.socket?.emit('prepareResults', {
        action: 'redirect',
        url: `/results`,
        data: resultData
      });
    }

    this.io.to(roomId).emit('chatEnded');
    setTimeout(() => this.chatRooms.delete(roomId), 5000);
  }

  handleMessage(socket, roomId, message, senderName) {
    const room = this.chatRooms.get(roomId);
    if (!room) throw new Error('Sala não encontrada');
    const partnerId = room.pairs.get(socket.id);

    // Se não há par mapeado, está sem parceiro (ou bloqueado). Não envia.
    if (!partnerId) {
      socket.emit('partnerBlocked', {
        message: 'Não é possível enviar mensagens nesta fase (sem parceiro ou parceiro bloqueado).'
      });
      return;
    }

    const partnerSocket = this.io.sockets.sockets.get(partnerId);
    if (!partnerSocket) {
      socket.emit('partnerBlocked', {
        message: 'Seu parceiro está offline no momento.'
      });
      return;
    }

    // Verifica bloqueio por garantia
    const everyone = Array.from(room.participants.values());
    const me = everyone.find(p => p.socketId === socket.id);
    const partner = everyone.find(p => p.socketId === partnerId);

    const aId = toNumericId(me?.userInfo?.userId || me?.userId);
    const bId = toNumericId(partner?.userInfo?.userId || partner?.userId);

    Users.isBlockedBetween(aId, bId).then(blocked => {
      if (blocked) {
        socket.emit('partnerBlocked', {
          message: 'Conversa bloqueada entre vocês. Aguarde o fim da fase.'
        });
        return;
      }

      const nowIso = new Date().toISOString();
      partnerSocket.emit('newMessage', {
        sender: socket.id,
        message,
        senderName: senderName || 'Parceiro',
        timestamp: nowIso,
        isSelf: false
      });

      // ACK visual para quem enviou (com timestamp consistente)
      socket.emit('messageAck', {
        message,
        timestamp: nowIso
      });
    }).catch(() => {
      // Em caso de erro na checagem, por segurança não envia
      socket.emit('partnerBlocked', {
        message: 'Não foi possível validar o envio agora.'
      });
    });
  }

  handleDisconnect(socketId) {
    this.chatRooms.forEach((room, roomId) => {
      let disconnectedUser = null;

      for (const [sessionId, participant] of room.participants) {
        if (participant.socketId === socketId) {
          disconnectedUser = participant;
          break;
        }
      }

      if (disconnectedUser) {
        const partnerId = room.pairs.get(socketId);
        if (partnerId) {
          const partnerSocket = this.io.sockets.sockets.get(partnerId);
          if (partnerSocket) {
            partnerSocket.emit('partnerDisconnected', {
              message: 'Seu parceiro desconectou. Aguardando reconexão...',
              canReconnect: true
            });
          }
        }

        disconnectedUser.socket = null;
        disconnectedUser.socketId = null;

        setTimeout(() => {
          if (!disconnectedUser.socketId) {
            room.participants.delete(disconnectedUser.sessionId);
            room.pairs.delete(socketId);
            this.io.to(roomId).emit('userLeftPermanently', {
              sessionId: disconnectedUser.sessionId
            });
          }
        }, room.graceReconnectMs || 15000);
      }
    });
  }
}

module.exports = new ChatController();
