const authService = require('../services/authService');
const path = require('path');
const logger = require('../config/logger');
const production = false;
require('dotenv').config();

// ===== Config derivada do ENV =====
const ACCESS_TTL = process.env.ACCESS_TTL || '15m';
const REFRESH_TTL_DAYS = Number(process.env.REFRESH_TTL_DAYS || 7);

// Converte '15m' | '60m' | '2h' | '1d' para milissegundos
function ttlToMs(ttl) {
  if (!ttl || typeof ttl !== 'string') return 15 * 60 * 1000; // fallback 15m
  const m = ttl.match(/^(\d+)\s*([smhd])$/i);
  if (!m) return 15 * 60 * 1000;
  const n = Number(m[1]);
  const unit = m[2].toLowerCase();
  const mult = unit === 's' ? 1000
            : unit === 'm' ? 60 * 1000
            : unit === 'h' ? 60 * 60 * 1000
            :                 24 * 60 * 60 * 1000; // 'd'
  return n * mult;
}

const ACCESS_COOKIE_MS  = ttlToMs(ACCESS_TTL);
const REFRESH_COOKIE_MS = REFRESH_TTL_DAYS * 24 * 60 * 60 * 1000;

exports.showLogin = (req, res) => {
    res.sendFile(path.join(__dirname, '/../views/login.html'));
};

exports.showRegister = (req, res) => {
    res.sendFile(path.join(__dirname, '/../views/register.html'));
};

exports.showProfile = (req, res) => {
    res.sendFile(path.join(__dirname, '/../views/profile.html'));
};

exports.login = async (req, res, next) => {
    try {
        const { email, password } = req.body;
        const ctx = { ip: req.ip, userAgent: req.get('user-agent') };
        const { accessToken, refreshToken, user } = await authService.login(email, password, ctx);

        const cookieBase = {
            httpOnly: true,
            secure: process.env.NODE_ENV === 'production',
            sameSite: 'lax',
            path: '/'
        };

        res.cookie('token', accessToken, { ...cookieBase, maxAge: ACCESS_COOKIE_MS });
        res.cookie('rt', refreshToken, { ...cookieBase, maxAge: REFRESH_COOKIE_MS });

        res.json({ success: true, user });
    } catch (error) { next(error); }
};

exports.refresh = async (req, res, next) => {
    try {
        const oldRt = req.cookies.rt;
        const ctx = { ip: req.ip, userAgent: req.get('user-agent') };
        const { accessToken, refreshToken } = await authService.refresh(oldRt, ctx);

        const cookieBase = {
            httpOnly: true,
            secure: process.env.NODE_ENV === 'production',
            sameSite: 'lax',
            path: '/'
        };
        res.cookie('token', accessToken, { ...cookieBase, maxAge: ACCESS_COOKIE_MS });
        res.cookie('rt', refreshToken, { ...cookieBase, maxAge: REFRESH_COOKIE_MS });

        res.json({ success: true });
    } catch (error) { next(error); }
};

exports.logout = async (req, res) => {
    const rt = req.cookies.rt;
    await authService.logout(rt);

    const opts = {
        httpOnly: true,
        secure: process.env.NODE_ENV === 'production',
        sameSite: 'lax',
        path: '/'
    };
    res.clearCookie('token', opts);
    res.clearCookie('rt', opts);
    res.json({ success: true });
};

exports.logoutAll = async (req, res, next) => {
    try {
        const at = req.cookies.token;
        const user = await authService.getUserData(at);
        await authService.logoutAll(user.id);

        const opts = {
            httpOnly: true,
            secure: process.env.NODE_ENV === 'production',
            sameSite: 'lax',
            path: '/'
        };
        res.clearCookie('token', opts);
        res.clearCookie('rt', opts);
        res.json({ success: true });
    } catch (e) {
        next(e);
    }
};

exports.register = async (req, res, next) => {
    try {
        const { username, email, password, gender } = req.body;
        const userId = await authService.register(username, email, password, gender);
        res.status(201).json({ success: true, userId });
    } catch (error) {
        next(error);
    }
};

exports.getUserData = async (req,res,next)=>{
  try{
    const token = req.cookies.token;
    const user = await authService.getUserData(token);
    res.json(user);
  }catch(e){ next(e); }
};

exports.profileCompletion = async (req, res, next) => {
  try {
    const token = req.cookies.token;
    const payload = await authService.profileCompletion(token);
    res.json({ success: true, ...payload });
  } catch (e) { next(e); }
};

exports.updateProfile = async (req,res,next)=>{
  try{
    const token = req.cookies.token;
    const updated = await authService.updateProfile(token, req.body, req.file);
    res.json({ success:true, user: updated });
  }catch(e){ next(e); }
};
exports.removeProfilePicture = async (req,res,next)=>{
  try{
    const token = req.cookies.token;
    await authService.removeProfilePicture(token);
    res.json({ success:true });
  }catch(e){ next(e); }
};

exports.updateInterests = async (req,res,next)=>{
  try{
    const token = req.cookies.token;
    const { interests } = req.body; // array de slugs
    await authService.updateUserInterests(token, interests);
    res.json({ success:true });
  }catch(e){ next(e); }
};

exports.profileOptions = async (_req,res,_next)=>{
  // opcional: servir as opções do backend (mantemos no front também)
  res.json({
    success:true,
    data:{ /* poderia devolver mesmas OPTIONS do front para centralizar */ }
  });
};

// Galeria
exports.uploadGalleryImage = async (req, res, next) => {
    try {
        const token = req.cookies.token;
        const image = await authService.uploadGalleryImage(token, req.file);
        res.json({ success: true, image });
    } catch (error) {
        next(error);
    }
};

exports.getGalleryImages = async (req, res, next) => {
    try {
        const token = req.cookies.token;
        const images = await authService.getGalleryImages(token);
        res.json({ success: true, images });
    } catch (error) {
        next(error);
    }
};

exports.deleteGalleryImage = async (req, res, next) => {
    try {
        const token = req.cookies.token;
        await authService.deleteGalleryImage(token, req.params.id);
        res.json({ success: true });
    } catch (error) {
        next(error);
    }
};

exports.listInterests = async (req, res, next) => {
  try {
    const items = await authService.listInterests();
    res.json({ success: true, items });
  } catch (e) { next(e); }
};

// Upload múltiplo para galeria
exports.uploadGalleryImages = async (req, res, next) => {
  try {
    const token = req.cookies.token;
    const images = await authService.uploadGalleryImages(token, req.files || []);
    res.json({ success: true, images });
  } catch (error) { next(error); }
};